define('quick-edit/create-linked-issue/create-linked-issue-form', [
    'quick-edit/form/configurable/create-issue',
    'quick-edit/form/field/configurable',
    'quick-edit/templates/issue',
    'quick-edit/util/jira',
    'jira/issue',
    'jira/util/events',
    'jira/util/events/types',
    'jira/message',
    'wrm/context-path',
    'jquery'
], function(
    ConfigurableCreateIssueForm,
    ConfigurableField,
    IssueTemplates,
    JiraUtil,
    JIRAIssue,
    JIRAEvents,
    JIRAEventTypes,
    JiraMessage,
    contextPath,
    jQuery
){
    'use strict';

    /**
     * A View that renders the Create linked issue form
     *
     * @class CreateLinkedIssueForm
     * @extends ConfigurableCreateIssueForm
     */
    return ConfigurableCreateIssueForm.extend({

        init: function (options) {
            ConfigurableCreateIssueForm.prototype.init.apply(this, arguments);
            this.action = contextPath() + "/secure/QuickCreateLinkedIssue.jspa?decorator=none&sourceIssueKey=" + JIRAIssue.getIssueKey();
        },

        getActiveFieldIds: function () {
            var ids = ConfigurableCreateIssueForm.prototype.getActiveFieldIds.apply(this, arguments);

            // These fields go together
            if (ids.indexOf("issuelinks")) {
                ids.push("issuelinks-linktype");
                ids.push("issuelinks-issues");
            }
            return ids;
        },

        // We want to display Issue Links, Summary and Description first
        reorderFieldsToRender: function(fields) {
            var reorderedFields = new Array(3);

            jQuery.each(fields, function () {
                var fieldId = this.getId();

                if (fieldId === "issuelinks") {
                    reorderedFields[0] = this;
                } else if (fieldId === "summary") {
                    reorderedFields[1] = this;
                } else if (fieldId === "description") {
                    reorderedFields[2] = this;
                } else {
                    reorderedFields.push(this);
                }
            });

            return reorderedFields;
        },

        createField: function (descriptor) {
            return new ConfigurableField(descriptor);
        },

        handleSubmitSuccess: function (data) {
            this.helper.handleSubmitSuccess(data);

            var html = JIRAIssue.issueCreatedMessage(data);

            jQuery.each(this.model.getPostIssueCreationErrorMessages(), function() {
                html = html.concat("<p>" + this);
            });

            JiraMessage.showSuccessMsg(html, {
                closeable: true
            });

            // need to refresh the issue view to reload the linked issues and see what we just did
            JIRAEvents.trigger(JIRAEventTypes.REFRESH_ISSUE_PAGE, [JIRAIssue.getIssueId()]);
        },

        _render: function () {
            var deferred = jQuery.Deferred(),
                instance = this;

            instance.model.getIssueSetupFields().done(function (issueSetupFields) {
                var html = IssueTemplates.createIssueForm({
                    issueSetupFields: issueSetupFields,
                    atlToken: instance.model.getAtlToken(),
                    formToken: instance.model.getFormToken(),
                    title: instance.title,
                    modifierKey: JiraUtil.getModifierKey(),
                    disallowCreateAnother: true
                });

                // add form chrome to container element
                instance.$element.html(html);

                instance.helper.decorate();

                // render fields
                instance.renderFormContents().done(function (el, scripts) {
                    instance.helper.restoreFormDirtyStatus();
                    instance.helper.decorateFields();
                    instance._decorateIssueLinksField();

                    deferred.resolveWith(instance, [instance.$element, scripts]);
                });
            });

            return deferred.promise();
        },

        _decorateIssueLinksField: function () {
            this.$element.find("label[for='issuelinks-linktype']").text(AJS.I18n.getText('createlinkedissue.issuelinks.created.issue'));
            this.$element.find("label[for='issuelinks-issues']").text(AJS.I18n.getText('createlinkedissue.issuelinks.linked.issues'));
            this.$element.find("#qf-field-issuelinks .description").text(AJS.I18n.getText('createlinkedissue.issuelinks.help'));
        }
    });
});
