AJS.test.require(["com.atlassian.jira.jira-projects-plugin:feature-discovery-test"], function(){
    "use strict";

    var jQuery = require("jquery");
    var wrmContextPath = require("wrm/context-path");
    var ProjectsAPI = require("jira/api/projects");
    var SidebarAPI = require("jira/api/projects/sidebar");
    var FeatureDiscovery = require("jira/projects/sidebar/feature-discovery");

    module("jira/projects/sidebar/feature-discovery", {

        setup: function() {
            this.sandbox = sinon.sandbox.create();
            this.server = sinon.fakeServer.create();
        },

        teardown: function() {
            this.server.restore();
            this.sandbox.restore();
        },

        mockCurrentProject: function(projectKey) {
            this.sandbox.stub(ProjectsAPI, "getCurrentProjectKey").returns(projectKey);
        },

        mockServerResponse: function(projectKey, badgesToRemove) {
            this.server.respondWith(
                "PUT",
                    wrmContextPath() + '/rest/projects/1.0/project/' + projectKey + '/badges',
                [
                    200,
                    { "Content-Type": "application/json" },
                    JSON.stringify({ badgesToRemove: badgesToRemove})
                ]
            );
        },

        mockNavigationItemWithId: function(navigationItemId) {
            var navigationItem = { removeBadge: function() {} };
            var mockSidebarComponent = this.mockSidebarApi();
            mockSidebarComponent.getItem.withArgs(navigationItemId).returns(navigationItem);
            return navigationItem;
        },

        mockNonExistentNavigationItemWithId: function(navigationItemId) {
            var mockSidebarComponent = this.mockSidebarApi();
            mockSidebarComponent.getItem.withArgs(navigationItemId).returns(undefined);
        },

        mockSidebarApi: function() {
            var mockSidebarComponent = {
                getItem: this.sandbox.stub()
            };
            var mockPromise = new jQuery.Deferred().resolve(mockSidebarComponent);
            this.sandbox.stub(SidebarAPI, 'getSidebar').returns(mockPromise);
            return mockSidebarComponent;
        }
    });

    test("Should remove the badges for the navigation items sent by the server in the response", function() {
        var projectKey = "KEY";
        this.mockCurrentProject(projectKey);

        var badgesToRemove = ["navigationItem"];
        this.mockServerResponse(projectKey, badgesToRemove);

        var navigationItem = this.mockNavigationItemWithId("navigationItem");
        var removeBadge = this.sandbox.spy(navigationItem, "removeBadge");

        FeatureDiscovery.updateBadges();
        this.server.respond();

        sinon.assert.calledOnce(removeBadge);
    });

    test("Should not execute badges request if no project key is defined", function() {
        this.mockCurrentProject(undefined);

        this.sandbox.spy(jQuery, "ajax");
        FeatureDiscovery.updateBadges();

        sinon.assert.notCalled(jQuery.ajax);
    });

    test("Should do nothing on the navigation items sent by the server if they don't exist", function() {
        var projectKey = "KEY";
        var fakeNavId = "navigationItem";
        var badgesToRemove = [fakeNavId];

        this.mockCurrentProject(projectKey);
        this.mockServerResponse(projectKey, badgesToRemove);

        this.mockNonExistentNavigationItemWithId(fakeNavId);

        FeatureDiscovery.updateBadges();
        this.server.respond();

        ok(true, "Non existent navigation items have been handled properly, since no exceptions have occurred");
    });
});

