AJS.test.require(['com.atlassian.jira.jira-projects-plugin:release-page-assets'], function(){
    "use strict";

    var jQuery = require('jquery');
    var Deferred = require("jira/jquery/deferred");
    var Backbone = require("jira-projects-backbone");
    var Marionette = require('jira/projects/libs/marionette');
    var SubmitModelStates = require("jira/projects/abstract-model/submit-model-states");

    var NavigationUtils = require("jira/projects/abstract-list/navigation-utils");

    function generateVersion(name, released) {
        function generateStatus() {
            return {
                count: (Math.floor(Math.random() * 3) == 0) ? 0 : Math.floor(Math.random() * 10) + 1,
                jqlUrl: "/jira/secure/IssueNavigator.jspa?reset=true&jqlQuery=fixVersion+%3D+%22" + encodeURIComponent(name) + "%22"
            }
        }
        return {
            name: "Version " + name,
            url: "/jira/browse/SSP/fixforversion/10000",
            released: released,
            startDate: { formatted: "27/Jun/14", iso: "2014-06-27" },
            releaseDate:{ formatted: (released) ? "27/Jul/14" : "", iso: (released) ? "2014-07-27" : "" },
            description: "Description " + name,
            operations: [],
            status: {
                unmapped: generateStatus(),
                toDo: generateStatus(),
                inProgress: generateStatus(),
                complete: generateStatus()
            }
        }
    }

    function generateVersionArray(length, unreleasedOnly) {
        var versions = [];
        for (var i = 1; i <= length; i++) {
            versions.push(generateVersion("" + i, (unreleasedOnly) ? false : (i % 2 === 0)));
        }
        return versions;
    }

    module('jira/projects/page/release/release-page-app', {
        setup: function () {
            this.context = AJS.test.mockableModuleContext();
            this.sandbox = sinon.sandbox.create();
            this.mockModules();

            this.container = jQuery(
                    '<div id="release-page">' +
                    '    <div id="filter-container"></div>' +
                    '    <div id="content-container"></div>' +
                    '</div>'
            );
            this.options = {
                container: this.container
            };

            this.projectsApi.getCurrentProjectId.returns("MON");
            this.storageStatusKey = "jira-projects:release-page-status-MON";

            this.app = this.context.require("jira/projects/page/release/release-page-app");
        },
        teardown: function () {
            this.sandbox.restore();
            Backbone.History.started = false;
        },
        mockModules: function () {
            this.context.mock("jira/projects/abstract-list/navigation-utils", NavigationUtils);
            this.sandbox.stub(NavigationUtils, "navigate");
            this.sandbox.stub(NavigationUtils, "append");
            this.sandbox.stub(NavigationUtils, "getQueryStringParam");

            this.localStorage = {
                getItem: sinon.stub(),
                setItem: sinon.stub(),
                removeItem: sinon.stub()
            };
            this.context.mock("jira/data/local-storage", this.localStorage);

            this.projectsApi = {
                getCurrentProjectId: sinon.stub(),
                getCurrentProjectKey: sinon.stub()
            };
            this.context.mock("jira/api/projects", this.projectsApi);
            this.context.mock("jira/projects/libs/marionette", Marionette);
            this.calendarData = {
                ifFormat: "%e/%b/%y"
            };
            this.context.mock("jira/projects/release/calendar-params-data", this.calendarData);

            this.backbone = jQuery.extend({}, Backbone, {
                History: {
                    started: false
                }
            });
            this.context.mock("jira-projects-backbone", this.backbone);

            this.wrmData = {
                claim: sinon.stub()
            };
            this.wrmData.claim.returns({});
            this.context.mock("wrm/data", this.wrmData);

            this.featureManager = {
                isFeatureEnabled: this.sandbox.stub()
            };
            this.context.mock("jira/featureflags/feature-manager", this.featureManager);
        },
        triggerContainsChange: function () {
            this.sandbox.useFakeTimers();
            this.container.find('#version-filter-text').trigger("input");
            this.sandbox.clock.tick(300);
        },
        stubData: function stubData(data) {
            this.options.data = data;
        },
        mockedModel: function() {
            return new Backbone.Model();
        }
    });

    test('Should filter when changing criteria', function () {
        this.stubData(generateVersionArray(20));
        this.app.start(this.options);

        equal(this.container.find('#versions-table tbody.items tr:first td:eq(2)').text(), "common.filters.versionstatus.unreleased", "First version on the list is unreleased.");
        equal(this.container.find('#versions-table tbody.items tr:eq(1) td:eq(2)').text(), "common.filters.versionstatus.unreleased", "Second version on the list is unreleased.");

        this.container.find('.version-filter-status-released').click();
        this.container.find('.version-filter-status-unreleased').click();

        equal(this.container.find('#versions-table tbody.items tr:first td:eq(2)').text(), "common.filters.versionstatus.released", "First version on the list is released.");
        equal(this.container.find('#versions-table tbody.items tr:eq(1) td:eq(2)').text(), "common.filters.versionstatus.released", "Second version on the list is released.");


        this.container.find('#version-filter-text').val("Version 4");
        this.triggerContainsChange();

        equal(this.container.find('#versions-table tbody.items tr:first td:eq(1)').text(), "Version 4");
        equal(this.container.find('#versions-table tbody.items tr').length, 1);
    });

    test('Should display message when no versions match filter criteria', function () {
        this.stubData(generateVersionArray(10));
        this.app.start(this.options);

        this.container.find('#version-filter-text').val("asdf");
        this.triggerContainsChange();

        equal(this.container.find('#versions-table tbody.items tr').length, 1);
        equal(this.container.find('#versions-table .no-results').length, 1);
    });

    test('Should not change URL and use defaults if status is not in URL and not in local storage.', function () {
        this.stubData(generateVersionArray(10));
        this.app.start(this.options);

        equal(this.app.filter.get("status"), "unreleased");
        ok(!NavigationUtils.append.called);
    });

    test('Should use parameters in local storage if page does not have url', function () {
        this.stubData(generateVersionArray(10));
        this.localStorage.getItem.withArgs(this.storageStatusKey).returns("released");

        this.app.start(this.options);

        ok(NavigationUtils.append.calledOnce);
        ok(NavigationUtils.append.calledWith("status", "released"));
    });

    test('Should use URL even if some data are saved in local storage', function () {
        this.stubData(generateVersionArray(10));
        this.localStorage.getItem.withArgs(this.storageStatusKey).returns("released");
        NavigationUtils.getQueryStringParam.withArgs("status").returns("all");

        this.app.start(this.options);

        equal(this.app.filter.get("status"), "all");
        ok(!NavigationUtils.append.called);
    });

    test('Should save status in local storage on change of status', function () {
        this.stubData(generateVersionArray(10));

        this.app.start(this.options);
        this.app.filter.set("status", "all");

        ok(this.localStorage.setItem.calledWith(this.storageStatusKey, "all"));
    });

    test('Should not use parameter in local storage if assigned to other project', function () {
        this.stubData(generateVersionArray(10));
        this.localStorage.getItem.withArgs(this.storageStatusKey).returns("released");
        this.projectsApi.getCurrentProjectId.returns("MKY");

        this.app.start(this.options);

        equal(this.app.filter.get("status"), "unreleased");
        ok(!NavigationUtils.append.called);
    });

    test('Should trigger progressDialogClicked when progressDialogClicked is triggered from mainRegion.currentView', function () {
        this.stubData(generateVersionArray(50));
        this.app.start(this.options);
        var eventHandler = sinon.spy();
        this.app.on('progressDialogClicked', eventHandler);

        this.app.mainRegion.currentView.trigger('progressDialogClicked');

        sinon.assert.calledOnce(eventHandler);
    });

    test('Should trigger navigate when navigate is triggered from mainRegion.currentView', function () {
        this.stubData(generateVersionArray(50));
        this.app.start(this.options);
        var eventHandler = sinon.spy();
        this.app.on('navigate', eventHandler);

        this.app.mainRegion.currentView.trigger('navigate');

        sinon.assert.calledOnce(eventHandler);
    });

    test('Should break a very long component name', function () {
        var longName = "A veryveryveryveryveryvery veryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryver yveryveryveryvery veryveryveryveryveryveryveryveryveryvery longname";
        this.stubData([generateVersion(longName, false)]);
        this.app.start(this.options);

        // fix headers to some actual translation
        this.container.find('#versions-table tbody.header th, #versions-table tbody.items td.versions-table__status span').each(function(idx, el) {
            el.innerHTML = {
                "common.concepts.version": "Version",
                "common.words.status": "Status",
                "common.concepts.progress.bar": "Progress",
                "version.startdate": "Start date",
                "version.releasedate": "Release date",
                "common.concepts.description": "Description",
                "common.words.actions": "Actions",
                "common.filters.versionstatus.unreleased": "UNRELEASED"
            }[el.innerHTML];
        });

        //default is 1000 and column does not expand to 50%
        jQuery('#qunit-fixture').width(1500);
        // otherwise width/height is 0
        this.container.appendTo(jQuery('#qunit-fixture'));

        var bodyWidth = this.container.find('#versions-table tbody.items').width();
        var description = this.container.find('#versions-table tbody.items').find('.versions-table__description').width();

        var epsilon = (description / bodyWidth) * 100 - 50;
        equal(Math.abs(epsilon) < 15, true, "ABS(Epsilon) should be below 15, actual: " + epsilon);
    });
});

