AJS.test.require(['com.atlassian.jira.jira-projects-plugin:components-page-assets'], function () {
    "use strict";

    var _ = require("underscore");
    var contextPath = require("wrm/context-path")();
    var SubmitModelStates = require("jira/projects/abstract-model/submit-model-states");
    var $ = require("jquery");

    var Deferred = require("jira/jquery/deferred");
    var Backbone = require("jira-projects-backbone");

    module('SubmitModel', {
        setup: function () {
            this.context = AJS.test.mockableModuleContext();
            this.sandbox = sinon.sandbox.create();
            this.sandbox.useFakeServer();

            this.flags = {
                showErrorMsg: this.sandbox.stub()
            };
            this.context.mock("jira/flag", this.flags);
            this.context.mock("jira-projects-backbone", Backbone);

            this.SubmitModel = this.context.require("jira/projects/components/submitmodel");
            this.xhr = {};
        },
        teardown: function () {
            this.sandbox.restore();
        },

        defaultParameters: function (custom) {
            return _.extend({}, {
                "project": "1",
                "description": "Some description",
                "leadUserName": "kevin",
                "name": "component",
                "assigneeType": "type"
            }, custom);
        }
    });

    test("Save method should send only custom parameters", function () {
        var submitModel = new this.SubmitModel(this.defaultParameters({
            "other": "value"
        }));

        submitModel.save();

        this.sandbox.server.respond();

        var request = this.sandbox.server.requests[0];
        equal(request.url, contextPath + "/rest/api/2/component");
        deepEqual(JSON.parse(request.requestBody), this.defaultParameters());
    });

    test("Details should set all passed parameters and set state to ready", function(){
        var submitModel = new this.SubmitModel({
            id: 13,
            project: 7
        });
        var response = {
            "some": "value",
            "other": 10
        };

        // we are adding ?_=[0-9] to the end of url - that's why I'm using regex here
        this.sandbox.server.respondWith(/.*\/rest\/projects\/1.0\/project\/7\/component\/13\?_=.*/, [200, {"Content-Type": "application/json"}, JSON.stringify(response)]);

        submitModel.detailsSync();
        this.sandbox.server.respond();

        deepEqual(submitModel.attributes, _.extend({
            id: 13,
            project: 7,
            issuesCount: 0,
            archived: false,
            state: SubmitModelStates.READY
        }, response));
    });

    test("failed save() should set state to SubmitModelStates.ERROR_ADD", function () {
        var submitModel = new this.SubmitModel({});
        var response = {
            "errorMessages": [],
            "errors": {"name": "The component name specified is invalid - cannot be an empty string."}
        };
        this.sandbox.server.respondWith(/.*\/rest\/projects\/1.0\/project\/7\/component\/13\?_=.*/, [400, {"Content-Type": "application/json"}, JSON.stringify(response)]);

        submitModel.save();
        this.sandbox.server.respond();

        equal(submitModel.get('state'), SubmitModelStates.ERROR_ADD);
    });

    test("destroy() should set lastDestroyOptions on model object", function () {
        var submitModel = new this.SubmitModel({});
        var options = {data: {moveIssuesTo: 0}, someProperty: "1"};
        var deferred = new Deferred();
        var destroy = this.sandbox.stub(Backbone.Model.prototype, "destroy");
        destroy.returns(deferred);
        submitModel.destroy(_.clone(options));

        deepEqual(submitModel.lastDestroyOptions, options);
    });

    test("failed destroy() should set state to SubmitModelStates.ERROR_DELETE and errorMsg", function () {
        var submitModel = new this.SubmitModel({});
        var options = {data: {moveIssuesTo: 0}, someProperty: "1"};

        var deferred = new Deferred();
        var destroy = this.sandbox.stub(Backbone.Model.prototype, "destroy");
        destroy.returns(deferred);
        var errorMsg = "some error";
        this.sandbox.stub(AJS, "escapeHtml").returns(errorMsg);

        submitModel.destroy(options);
        deferred.reject(this.xhr);

        equal(submitModel.get('state'), SubmitModelStates.ERROR_DELETE);
        equal(submitModel.get('errorMsg'), errorMsg);
    });

    test("failed destroy() should not show any flag message error", function () {
        var submitModel = new this.SubmitModel({});
        var options = {data: {moveIssuesTo: 0}, someProperty: "1"};

        var deferred = new Deferred();
        var destroy = this.sandbox.stub(Backbone.Model.prototype, "destroy");
        destroy.returns(deferred);
        var errorMsg = "some error";
        this.sandbox.stub(AJS, "escapeHtml").returns(errorMsg);

        submitModel.destroy(options);
        deferred.reject(this.xhr);

        equal(this.flags.showErrorMsg.callCount, 0);
    });
});

