/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.index.ha;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.jira.cluster.disasterrecovery.JiraHomeChangeEvent;
import com.atlassian.jira.index.LuceneVersion;
import com.atlassian.jira.index.ha.IndexSnapshotContribution;
import com.atlassian.jira.issue.index.IssueIndexManager;
import com.atlassian.jira.util.LuceneDirectoryUtils;
import com.atlassian.jira.util.PathUtils;
import com.atlassian.jira.util.TempDirectoryUtil;
import com.atlassian.jira.util.ZipUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.io.PatternFilenameFilter;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.comparator.LastModifiedFileComparator;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.NoSuchDirectoryException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexUtils {
    private static final Logger log = LoggerFactory.getLogger(IndexUtils.class);
    private static final String INDEX_SNAPSHOT_PREFIX = "IndexSnapshot_";
    private static final String INDEX_SNAPSHOT_EXT = ".zip";
    private static final Pattern INDEX_SNAPSHOT_PATTERN = Pattern.compile(Pattern.quote("IndexSnapshot_") + ".*" + Pattern.quote(".zip"));
    public static final PatternFilenameFilter INDEX_SNAPSHOT_FILTER = new PatternFilenameFilter(INDEX_SNAPSHOT_PATTERN);
    private final IssueIndexManager issueIndexManager;
    private final LuceneDirectoryUtils luceneDirectoryUtils;
    private final EventPublisher eventPublisher;

    public IndexUtils(IssueIndexManager issueIndexManager, LuceneDirectoryUtils luceneDirectoryUtils, EventPublisher eventPublisher) {
        this.issueIndexManager = issueIndexManager;
        this.luceneDirectoryUtils = luceneDirectoryUtils;
        this.eventPublisher = eventPublisher;
    }

    public String takeIndexSnapshot(@Nonnull String sourcePath, @Nonnull String destinationPath, @Nonnull String snapshotId, int maxSnapshots, @Nullable IndexSnapshotContribution contribution) {
        File workDir = TempDirectoryUtil.createTempDirectory("JIRAIndexBackup");
        try {
            log.debug("Copying indexes for snapshot");
            this.copyIndexes(sourcePath, workDir.getCanonicalPath());
            if (contribution != null) {
                contribution.writeContribution(workDir);
            }
            String filename = INDEX_SNAPSHOT_PREFIX + snapshotId + INDEX_SNAPSHOT_EXT;
            log.debug("Preparing snapshot file {}", (Object)filename);
            File destination = new File(destinationPath);
            if (!destination.exists()) {
                destination.mkdir();
            }
            File snapshot = new File(destination, filename);
            ZipUtils.zip(workDir, snapshot);
            this.eventPublisher.publish((Object)new JiraHomeChangeEvent(JiraHomeChangeEvent.Action.FILE_ADD, JiraHomeChangeEvent.FileType.INDEX_SNAPSHOT, snapshot));
            log.debug("Deleting old snapshot files");
            this.deleteOldSnapshots(destination, maxSnapshots);
            log.debug("Finished taking snapshot");
            String string = filename;
            return string;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        finally {
            FileUtils.deleteQuietly((File)workDir);
        }
    }

    public void copyIndexes(@Nonnull String sourcePath, @Nonnull String destinationPath) {
        Map<IndexPath, IndexPathMapping> indexPathMappings = this.buildIndexPathMappings(sourcePath, destinationPath);
        for (IndexPathMapping indexPathMapping : indexPathMappings.values()) {
            File sourceDirectory = new File(indexPathMapping.getSourcePath());
            File destDirectory = new File(indexPathMapping.getDestinationPath());
            this.copySpecificIndex(sourceDirectory, destDirectory);
        }
    }

    public Map<IndexPath, IndexPathMapping> buildIndexPathMappings(@Nonnull String sourcePath, @Nonnull String destinationPath) {
        HashMap indexPathMappings = Maps.newHashMap();
        for (IndexPath path : IndexPath.values()) {
            indexPathMappings.put(path, new IndexPathMapping(PathUtils.joinPaths((String[])new String[]{sourcePath, path.getPath()}), PathUtils.joinPaths((String[])new String[]{destinationPath, path.getPath()})));
        }
        return indexPathMappings;
    }

    public void clearIndex(@Nonnull String path) {
        List<String> indexPaths = this.buildIndexPaths(path);
        for (String indexPath : indexPaths) {
            this.clearIndex(new File(indexPath));
        }
    }

    private void clearIndex(File directory) {
        IndexWriter writer = null;
        try {
            writer = this.getWriter(directory);
            writer.deleteAll();
        }
        catch (Exception e) {
            log.error("Error occured while copying index", (Throwable)e);
            throw new RuntimeException(e);
        }
        finally {
            this.closeQuietly(writer);
        }
    }

    private List<String> buildIndexPaths(String path) {
        ArrayList paths = Lists.newArrayList();
        for (IndexPath indexPath : IndexPath.values()) {
            paths.add(PathUtils.joinPaths((String[])new String[]{path, indexPath.getPath()}));
        }
        return paths;
    }

    /*
     * Loose catch block
     */
    private void copySpecificIndex(File sourceDirectory, File destDirectory) {
        IndexReader reader = null;
        IndexWriter writer = null;
        try {
            writer = this.getWriter(destDirectory);
            reader = IndexReader.open((Directory)this.luceneDirectoryUtils.getDirectory(sourceDirectory));
            writer.addIndexes(new IndexReader[]{reader});
            this.closeQuietly(reader);
            this.closeQuietly(writer);
        }
        catch (NoSuchDirectoryException e) {
            log.debug("Cannot copy index; " + e.getMessage());
            this.closeQuietly(reader);
            this.closeQuietly(writer);
        }
        catch (Exception e2) {
            log.error("Error occured while copying index", (Throwable)e2);
            throw new RuntimeException(e2);
            {
                catch (Throwable throwable) {
                    this.closeQuietly(reader);
                    this.closeQuietly(writer);
                    throw throwable;
                }
            }
        }
    }

    private void closeQuietly(IndexWriter writer) {
        if (writer != null) {
            try {
                writer.close();
            }
            catch (IOException e) {
                log.debug("Exception thrown while closing writer, ignored");
            }
        }
    }

    private void closeQuietly(IndexReader reader) {
        if (reader != null) {
            try {
                reader.close();
            }
            catch (IOException e) {
                log.debug("Exception thrown while closing reader, ignored");
            }
        }
    }

    private IndexWriter getWriter(File directory) throws Exception {
        IndexWriterConfig indexWriterConfig = new IndexWriterConfig(LuceneVersion.get(), this.issueIndexManager.getAnalyzerForIndexing());
        indexWriterConfig.setOpenMode(IndexWriterConfig.OpenMode.CREATE);
        return new IndexWriter(this.luceneDirectoryUtils.getDirectory(directory), indexWriterConfig);
    }

    @VisibleForTesting
    protected int deleteOldSnapshots(File directory, int numToKeep) {
        File[] snapshots = directory.listFiles((FilenameFilter)INDEX_SNAPSHOT_FILTER);
        Arrays.sort(snapshots, LastModifiedFileComparator.LASTMODIFIED_REVERSE);
        int numKept = 0;
        int numDeleted = 0;
        for (File snapshot : snapshots) {
            if (numKept < numToKeep) {
                ++numKept;
                continue;
            }
            if (!snapshot.delete()) continue;
            this.eventPublisher.publish((Object)new JiraHomeChangeEvent(JiraHomeChangeEvent.Action.FILE_DELETED, JiraHomeChangeEvent.FileType.INDEX_SNAPSHOT, snapshot));
            ++numDeleted;
        }
        return numDeleted;
    }

    static class IndexPathMapping {
        private final String sourcePath;
        private final String destinationPath;

        IndexPathMapping(String sourcePath, String destinationPath) {
            this.sourcePath = sourcePath;
            this.destinationPath = destinationPath;
        }

        String getSourcePath() {
            return this.sourcePath;
        }

        String getDestinationPath() {
            return this.destinationPath;
        }
    }

    public static enum IndexPath {
        ISSUES("issues"),
        COMMENTS("comments"),
        CHANGE_HISTORY("changes"),
        WORKLOGS("worklogs"),
        SEARCH_REQUESTS(PathUtils.joinPaths((String[])new String[]{"entities", "searchrequest"})),
        PORTAL_PAGES(PathUtils.joinPaths((String[])new String[]{"entities", "portalpage"}));

        private final String path;

        private IndexPath(String path) {
            this.path = path;
        }

        public String getPath() {
            return this.path;
        }
    }
}

