/*
 * Copyright (c) 2002-2004
 * All rights reserved.
 */

package com.atlassian.jira.web.bean;

import com.atlassian.core.bean.MathBean;
import org.apache.log4j.Logger;

import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * This is used to store the statistics retrieved by the StatisticAccessorBean.
 * Formerly an inner class in StatisticAccessorBean. Should be moved out as it is accessed by tests
 * Since stats are accessed through this object, it should be made public to allow other classes access to stats
 * generated by the stat bean
 */
public class StatisticMapWrapper<K,N extends Number> extends AbstractMap<K,N>
{
    private static final Logger log = Logger.getLogger(StatisticMapWrapper.class);

    private final long totalcount;
    private long largestPercentage = -1;
    private Map<K,N> statistics;
    private Map<K,Long> percentages;
    private final long irrelevantCount;

    public StatisticMapWrapper(Map<K,N> statistics, long totalCount, long irrelevantCount)
    {
        this.totalcount = totalCount;
        this.irrelevantCount = irrelevantCount;
        setStatistics(statistics);
    }

    public Map<K,N> getStatistics()
    {
        return statistics;
    }

    protected Map<K,Long> getPercentages()
    {
        if (percentages == null)
        {
            percentages = new HashMap<K,Long>();
            for (K o : keySet())
            {
                long value = get(o).longValue();
                long percentage = (new MathBean()).getPercentage(value, getTotalCount());
                percentages.put(o, percentage);
            }
        }
        return percentages;
    }

    public int getIrrelevantPercentage()
    {
        return (int) (new MathBean()).getPercentage(irrelevantCount, getTotalCount());
    }

    public void setStatistics(Map<K,N> statistics)
    {
        this.statistics = statistics;
    }

    public Set<Entry<K,N>> entrySet()
    {
        return statistics.entrySet();
    }

    /**
     * Return the total number of issues that were matched.
     * @return total number of issues that were matched.
     */
    public long getTotalCount()
    {
        return this.totalcount;
    }

    public int getIrrelevantCount()
    {
        return (int) irrelevantCount;
    }

    public long getLargestPercentage()
    {
        if (largestPercentage == -1)
        {
            for (Long o : getPercentages().values())
            {
                long currentValue = longValue(o);
                if (largestPercentage < currentValue)
                {
                    largestPercentage = currentValue;
                }
            }
        }
        return largestPercentage;
    }

    public int getPercentage(K key)
    {
        try
        {
           return getPercentages().get(key).intValue();
        }
        catch (Exception e)
        {
            log.error("Error getting stats for key=" + key + ": " + e, e);
            return 0;
        }
    }

    // Retrieve the value for 'null key' entry in the map
    // For example, issues without a priority have a null key in the map
    public long getNullKeyValue()
    {
        boolean nullValueFound = false;
        long value = -1;

        for (K o : keySet())
        {
            if (o == null)
            {
                if (nullValueFound)
                {
                    // Should only be one mapping with a 'null' key - we got problems otherwise!
                    log.error("StatisticMap - contains multiple mappings with a null key");
                    return -1;
                } else
                {
                    value = get(o).longValue();
                    nullValueFound = true;
                }
            }
        }
        return value;
    }

    private long longValue(Number value)
    {
        return value.longValue();
    }
}
