/*
 * Created by IntelliJ IDEA.
 * User: Mike
 * Date: Oct 1, 2004
 * Time: 5:22:03 PM
 */
package com.atlassian.jira.util;

import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.datetime.LocalDate;
import com.atlassian.jira.datetime.LocalDateFactory;
import com.atlassian.jira.issue.index.IndexException;
import org.apache.log4j.Logger;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.chrono.ISOChronology;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;

import javax.annotation.concurrent.Immutable;
import java.io.File;
import java.io.IOException;
import java.util.Date;

/**
 * A simple utility class for our common Lucene usage methods.
 */
public class LuceneUtils
{
    private static final Logger log = Logger.getLogger(LuceneUtils.class);
    private static final DateFormatter dateFormatter = new DateFormatter();

    /**
     * do not construct
     */
    private LuceneUtils()
    {}

    public static String localDateToString(LocalDate localDate)
    {
        return LocalDateFactory.toIsoBasic(localDate);
    }

    public static LocalDate stringToLocalDate(final String indexValue)
    {
        return LocalDateFactory.fromIsoBasicFormat(indexValue);
    }

    /**
     * Turns a given date-time (point in time) value into a String suitable for storing and searching in Lucene.
     * <p>
     * The date-time is stored with second precision using GMT eg 2011-05-05 10:59:39.000 EST (GMT+10) would return "20110505005939".
     * Note that the lexigraphical ordering of such Strings are in line with the ordering of the represented timestamps.
     *
     * @param date the date to be converted
     * @return a string in format <code>YYYYMMDDHHMMSS</code> in GMT timezone.
     */
    public static String dateToString(final Date date)
    {
        return dateFormatter.dateToString(date, Resolution.SECOND);
    }

    public static Date stringToDate(final String s)
    {
        if ((s != null) && (s.trim().length() > 0))
        {
            return dateFormatter.stringToDate(s);
        }
        return new Date();
    }

    @Immutable
    static class DateFormatter
    {
        private final DateTimeFormatter year;
        private final DateTimeFormatter month;
        private final DateTimeFormatter day;
        private final DateTimeFormatter hour;
        private final DateTimeFormatter minute;
        private final DateTimeFormatter second;
        private final DateTimeFormatter millisecond;

        DateFormatter()
        {
            final DateTimeFormatterBuilder builder = new DateTimeFormatterBuilder()
            {
                @Override
                public DateTimeFormatter toFormatter()
                {
                    return super.toFormatter().withZone(DateTimeZone.UTC);
                }
            };
            ISOChronology chronology = ISOChronology.getInstance();
            year = builder.appendYear(4, 4).toFormatter().withChronology(chronology);
            month = builder.appendMonthOfYear(2).toFormatter().withChronology(chronology);
            day = builder.appendDayOfMonth(2).toFormatter().withChronology(chronology);
            hour = builder.appendHourOfDay(2).toFormatter().withChronology(chronology);
            minute = builder.appendMinuteOfHour(2).toFormatter().withChronology(chronology);
            second = builder.appendSecondOfMinute(2).toFormatter().withChronology(chronology);
            millisecond = builder.appendMillisOfSecond(3).toFormatter().withChronology(chronology);
        }

        /**
         * Converts a string produced by <code>timeToString</code> or
         * <code>dateToString</code> back to a time, represented as a
         * Date object.
         *
         * @param dateString the date string to be converted
         * @return the parsed time as a Date object
         * @throws DateParsingException if <code>dateString</code> is not in the
         *  expected format
         */
        public Date stringToDate(final String dateString) throws DateParsingException
        {
            return stringToDateTime(dateString.trim()).toDate();
        }

        /**
         * Converts a string produced by <code>timeToString</code> or
         * <code>dateToString</code> back to a time, represented as a
         * Date object.
         *
         * @param dateString the date string to be converted
         * @return the parsed time as a Date object
         * @throws DateParsingException if <code>dateString</code> is not in the
         *  expected format
         */
        public DateTime stringToDateTime(final String dateString) throws DateParsingException
        {
            switch (dateString.length())
            {
                case 4:
                    return year.parseDateTime(dateString);
                case 6:
                    return month.parseDateTime(dateString);
                case 8:
                    return day.parseDateTime(dateString);
                case 10:
                    return hour.parseDateTime(dateString);
                case 12:
                    return minute.parseDateTime(dateString);
                case 14:
                    return second.parseDateTime(dateString);
                case 17:
                    return millisecond.parseDateTime(dateString);

                default:
                    throw new DateParsingException(dateString);
            }
        }

        /**
         * Converts a Date to a string suitable for indexing.
         *
         * @param date the date to be converted
         * @param resolution the desired resolution, see
         *  {@link Resolution}
         * @return a string in format <code>yyyyMMddHHmmssSSS</code> or shorter,
         *  depending on <code>resolution</code>; using GMT as timezone
         */
        public String dateToString(final Date date, final Resolution resolution)
        {
            return timeToString(date.getTime(), resolution);
        }

        /**
         * Converts a millisecond time to a string suitable for indexing.
         *
         * @param date the date expressed as milliseconds since January 1, 1970, 00:00:00 GMT
         * @param resolution the desired resolution, see
         *  {@link Resolution}
         * @return a string in format <code>yyyyMMddHHmmssSSS</code> or shorter,
         *  depending on <code>resolution</code>; using GMT as timezone
         */
        public String timeToString(final long date, final Resolution resolution)
        {
            switch (resolution)
            {
                case YEAR:
                    return year.print(date);
                case MONTH:
                    return month.print(date);
                case DAY:
                    return day.print(date);
                case HOUR:
                    return hour.print(date);
                case MINUTE:
                    return minute.print(date);
                case SECOND:
                    return second.print(date);
                case MILLISECOND:
                    return millisecond.print(date);
            }
            throw new IllegalArgumentException("unknown resolution " + resolution);
        }
    }

    static class DateParsingException extends RuntimeException
    {
        public DateParsingException(final String dateString)
        {
            super("Input is not valid date string: " + dateString);
        }
    }

    private static enum Resolution
    {
        YEAR,
        MONTH,
        DAY,
        HOUR,
        MINUTE,
        SECOND,
        MILLISECOND
    }
}
