package com.atlassian.jira.issue.search.parameters.lucene.sort;

import com.atlassian.jira.issue.search.LuceneFieldSorter;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.search.FieldComparator;
import org.apache.lucene.search.FieldComparatorSource;

import java.io.IOException;
import java.util.Comparator;

/**
 * This Sort Comparator reads through the terms dictionary in lucene, and builds up a list of ordered terms.  It then
 * sorts the documents according to the order that they appear in the terms list.
 * <p/>
 * This approach, whilst very fast, does load the entire term dictionary into memory.  This could be a problem where
 * there are a large number of terms (eg. text fields).
 */
public class MappedSortComparator extends FieldComparatorSource
{
    private final LuceneFieldSorter sorter;

    public MappedSortComparator(LuceneFieldSorter sorter)
    {
        this.sorter = sorter;
    }

    @Override
    public FieldComparator newComparator(String fieldname, int numHits, int sortPos, boolean reversed)
            throws IOException
    {
        return new InternalFieldComparator(numHits, fieldname, sorter);
    }

    public final class InternalFieldComparator extends FieldComparator
    {
        private final Object[] values;
        private Object[] currentDocumentValues;
        private final String field;
        private LuceneFieldSorter sorter;
        private Object bottom;

        InternalFieldComparator(int numHits, String field, final LuceneFieldSorter sorter)
        {
            values = new Object[numHits];
            this.field = field;
            this.sorter = sorter;
        }

        @Override
        public int compare(int slot1, int slot2)
        {
            final Object v1 = values[slot1];
            final Object v2 = values[slot2];
            if (v1 == null)
            {
                if (v2 == null)
                {
                    return 0;
                }
                return 1;
            }
            else if (v2 == null)
            {
                return -1;
            }
            return sorter.getComparator().compare(v1, v2);
        }

        @Override
        public int compareBottom(int doc)
        {
            final Object v2 = currentDocumentValues[doc];
            if (bottom == null)
            {
                if (v2 == null)
                {
                    return 0;
                }
                return 1;
            }
            else if (v2 == null)
            {
                return -1;
            }
            return sorter.getComparator().compare(bottom, v2);
        }

        @Override
        public void copy(int slot, int doc)
        {
            values[slot] = currentDocumentValues[doc];
        }

        @Override
        public void setNextReader(IndexReader reader, int docBase) throws IOException
        {
            currentDocumentValues = getLuceneValues(field, reader);
        }

        @Override
        public void setBottom(final int bottom)
        {
            this.bottom = values[bottom];
        }

        @Override
        public Comparable<?> value(int slot)
        {
            // We won't be able to pull the values from the sort
            // This is only used by org.apache.lucene.search.FiledDoc instances, which we do not use.
            return null;
        }
    }

    /**
     * This makes a call into the JiraLuceneFieldFinder to retrieve values from the Lucence index.  It returns an array
     * that is the same size as the number of documents in the reader and will have all null values if the field is not
     * present, otherwise it has the values of the field within the document.
     * <p/>
     * Broken out as package level for unit testing reasons.
     *
     * @param field the name of the field to find
     * @param reader the Lucence index reader
     * @return an non null array of values, which may contain null values.
     * @throws IOException if stuff goes wrong
     */
    Object[] getLuceneValues(final String field, final IndexReader reader) throws IOException
    {
        return JiraLuceneFieldFinder.getInstance().getCustom(reader, field, MappedSortComparator.this);
    }

    /**
     * Returns an object which, when sorted according by the comparator returned from  {@link
     * LuceneFieldSorter#getComparator()} , will order the Term values in the correct order. <p>For example, if the
     * Terms contained integer values, this method would return <code>new Integer(termtext)</code>.  Note that this
     * might not always be the most efficient implementation - for this particular example, a better implementation
     * might be to make a ScoreDocLookupComparator that uses an internal lookup table of int.
     *
     * @param termtext The textual value of the term.
     * @return An object representing <code>termtext</code> that can be sorted by {@link
     *         LuceneFieldSorter#getComparator()}
     * @see Comparable
     * @see FieldComparator
     */
    public Object getComparable(String termtext)
    {
        return sorter.getValueFromLuceneField(termtext);
    }

    public Comparator getComparator()
    {
        return sorter.getComparator();
    }

    public boolean equals(Object o)
    {
        if (this == o)
        {
            return true;
        }
        if (o == null || getClass() != o.getClass())
        {
            return false;
        }

        final MappedSortComparator that = (MappedSortComparator) o;

        return (sorter == null ? that.sorter == null : sorter.equals(that.sorter));
    }

    public int hashCode()
    {
        return (sorter != null ? sorter.hashCode() : 0);
    }
}
