package com.atlassian.jira.issue.fields.rest.json.beans;

import com.atlassian.jira.config.properties.APKeys;
import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.link.IssueLinkManager;
import com.atlassian.jira.issue.link.IssueLinkType;
import com.atlassian.jira.issue.link.LinkCollection;
import com.atlassian.jira.security.JiraAuthenticationContext;
import com.google.common.collect.Lists;

import java.net.URI;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * Builder for {@link IssueLinkJsonBean} instances.
 *
 * @since v5.0
 */
public class IssueLinkBeanBuilder
{
    private final ApplicationProperties applicationProperties;
    private final IssueLinkManager issueLinkManager;
    private final JiraAuthenticationContext authContext;
    private final JiraBaseUrls jiraBaseUrls;
    private final Issue issue;

    public IssueLinkBeanBuilder(final ApplicationProperties applicationProperties, final IssueLinkManager issueLinkManager,
            final JiraAuthenticationContext authContext,
            JiraBaseUrls jiraBaseUrls, final Issue issue)
    {
        this.applicationProperties = applicationProperties;
        this.issueLinkManager = issueLinkManager;
        this.authContext = authContext;
        this.jiraBaseUrls = jiraBaseUrls;
        this.issue = issue;
    }

    /**
     * Build a List of IssueLinkJsonBean objects representing the issue links for the current Issue object.
     *
     * @return a List of IssueLinkJsonBean objects, or null if issue linking is disabled, or an empty List if no issue links
     *         exist
     */
    public List<IssueLinkJsonBean> buildIssueLinks()
    {
        if (!applicationProperties.getOption(APKeys.JIRA_OPTION_ISSUELINKING))
        {
            // issue linking disabled
            return null;
        }

        List<IssueLinkJsonBean> linkBeans = Lists.newArrayList();
        LinkCollection linkCollection = issueLinkManager.getLinkCollection(issue, authContext.getLoggedInUser());
        Set<IssueLinkType> linkTypes = linkCollection.getLinkTypes();
        if (linkTypes != null)
        {
            for (IssueLinkType issueLinkType : linkTypes)
            {
                List<Issue> outwardIssues = linkCollection.getOutwardIssues(issueLinkType.getName());
                if (outwardIssues != null)
                {
                    for (Issue issue : outwardIssues)
                    {
                        linkBeans.add(buildLink(issueLinkType, issue, true));
                    }
                }

                List<Issue> inwardIssues = linkCollection.getInwardIssues(issueLinkType.getName());
                if (inwardIssues != null)
                {
                    for (Issue issue : inwardIssues)
                    {
                        linkBeans.add(buildLink(issueLinkType, issue, false));
                    }
                }
            }
        }

        return linkBeans;
    }

    /**
     * Builds an IssueLinkJsonBean from an IssueLink.
     *
     * @param issueLinkType an IssueLinkType instance
     * @param issue an Issue that is linked to this.issue
     * @param isOutbound a boolean indicating whether it's an outbound link
     * @return an IssueLinkJsonBean
     */
    private IssueLinkJsonBean buildLink(IssueLinkType issueLinkType, Issue issue, boolean isOutbound)
    {
        URI linkTypeURI = URI.create(jiraBaseUrls.restApi2BaseUrl() + "issueLinkType/" + issueLinkType.getId());
        IssueLinkJsonBean issueLink = new IssueLinkJsonBean().type(IssueLinkTypeJsonBean.create(issueLinkType, linkTypeURI));

        return isOutbound ? issueLink.outwardIssue(createIssueLink(issue)) : issueLink.inwardIssue(createIssueLink(issue));
    }

    /**
     * Build an IssueLinkJsonBean object representing the parent link for the current Issue object.
     *
     * @return an IssueLinkJsonBean object, or null if no parent link exists
     */
    public IssueLinkJsonBean buildParentLink()
    {
        Issue parent = issue.getParentObject();
        if (parent == null)
        {
            return null;
        }

        return new IssueLinkJsonBean()
                .type(new IssueLinkTypeJsonBean().name("Parent"))
                .inwardIssue(createIssueLink(parent));
    }

    /**
     * Build a List of IssueLinkJsonBean objects representing the sub-task links for the current Issue object.
     *
     * @return a List of IssueLinkJsonBean objects, or an empty List if no sub-task links exist
     */
    public List<IssueLinkJsonBean> buildSubtaskLinks()
    {
        Collection<Issue> subtasks = issue.getSubTaskObjects();
        if (subtasks == null)
        {
            return Collections.emptyList();
        }

        List<IssueLinkJsonBean> subtaskLinks = Lists.newArrayListWithCapacity(subtasks.size());
        for (Issue subtask : subtasks)
        {
            subtaskLinks.add(new IssueLinkJsonBean()
                    .type(new IssueLinkTypeJsonBean().name("Sub-Task"))
                    .outwardIssue(createIssueLink(subtask))
            );
        }

        return subtaskLinks;
    }

    private IssueRefJsonBean createIssueLink(final Issue issue)
    {
        return new IssueRefJsonBean()
                .id(String.valueOf(issue.getId()))
                .key(issue.getKey())
                .self(URI.create(jiraBaseUrls.restApi2BaseUrl() + "issue/" + issue.getId()))
                .fields(new IssueRefJsonBean.Fields()
                        .summary(issue.getSummary())
                        .status(StatusJsonBean.shortBean(issue.getStatusObject(), jiraBaseUrls))
                        .issueType(IssueTypeJsonBean.shortBean(issue.getIssueTypeObject(), jiraBaseUrls))
                        .priority(PriorityJsonBean.shortBean(issue.getPriorityObject(), jiraBaseUrls))
                );
    }
}
