package com.atlassian.diagnostics.internal.platform.monitor.gc;

import com.atlassian.diagnostics.Severity;

import java.time.Duration;
import java.time.Instant;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

public class HighGCTimeDetails {
    private final Severity severity;
    private final String garbageCollectorName;
    private final Instant timestamp;
    private final Duration timeWindow;
    private final Duration garbageCollectionTime;
    private final long garbageCollectionCount;

    public HighGCTimeDetails(Severity severity,
                             String garbageCollectorName,
                             Instant timestamp,
                             Duration timeWindow,
                             Duration garbageCollectionTime,
                             long garbageCollectionCount) {
        this.severity = severity;
        this.garbageCollectorName = garbageCollectorName;
        this.timestamp = timestamp;
        this.timeWindow = timeWindow;
        this.garbageCollectionTime = garbageCollectionTime;
        this.garbageCollectionCount = garbageCollectionCount;
    }

    public static HighGCTimeAlertBuilder builder() {
        return new HighGCTimeAlertBuilder();
    }

    public Severity getSeverity() {
        return severity;
    }

    public String getGarbageCollectorName() {
        return garbageCollectorName;
    }

    public Instant getTimestamp() {
        return timestamp;
    }

    public Duration getTimeWindow() {
        return timeWindow;
    }

    public Duration getGarbageCollectionTime() {
        return garbageCollectionTime;
    }

    public long getGarbageCollectionCount() {
        return garbageCollectionCount;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        HighGCTimeDetails that = (HighGCTimeDetails) o;
        return garbageCollectionCount == that.garbageCollectionCount && severity == that.severity && garbageCollectorName.equals(that.garbageCollectorName) && timestamp.equals(that.timestamp) && timeWindow.equals(that.timeWindow) && garbageCollectionTime.equals(that.garbageCollectionTime);
    }

    @Override
    public int hashCode() {
        return Objects.hash(severity, garbageCollectorName, timestamp, timeWindow, garbageCollectionTime, garbageCollectionCount);
    }

    @Override
    public String toString() {
        return "HighGCTimeDetails{" +
                "severity=" + severity +
                ", garbageCollectorName='" + garbageCollectorName + '\'' +
                ", timestamp=" + timestamp +
                ", timeWindow=" + timeWindow +
                ", garbageCollectionTime=" + garbageCollectionTime +
                ", garbageCollectionCount=" + garbageCollectionCount +
                '}';
    }

    public static final class HighGCTimeAlertBuilder {
        private Severity severity;
        private String garbageCollectorName;
        private Instant timestamp;
        private Duration timeWindow;
        private Duration garbageCollectionTime;
        private Long garbageCollectionCount;

        private HighGCTimeAlertBuilder() {
        }

        public HighGCTimeAlertBuilder severity(Severity severity) {
            this.severity = severity;
            return this;
        }

        public HighGCTimeAlertBuilder garbageCollectorName(String garbageCollectorName) {
            this.garbageCollectorName = garbageCollectorName;
            return this;
        }

        public HighGCTimeAlertBuilder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public HighGCTimeAlertBuilder timeWindow(Duration timeWindow) {
            this.timeWindow = timeWindow;
            return this;
        }

        public HighGCTimeAlertBuilder garbageCollectionTime(Duration garbageCollectionTime) {
            this.garbageCollectionTime = garbageCollectionTime;
            return this;
        }

        public HighGCTimeAlertBuilder garbageCollectionCount(long garbageCollectionCount) {
            this.garbageCollectionCount = garbageCollectionCount;
            return this;
        }

        public HighGCTimeAlertBuilder addAlertInfo(GCDetailsCalculator gcDetailsCalculator) {
            this.garbageCollectionTime = gcDetailsCalculator.getTimeSpentOnCollection();
            this.timeWindow = gcDetailsCalculator.getTimeSinceLastPoll();
            this.garbageCollectionCount = gcDetailsCalculator.getCollectionCount();
            return this;
        }

        public HighGCTimeDetails build() {
            return new HighGCTimeDetails(
                    requireNonNull(severity),
                    requireNonNull(garbageCollectorName),
                    requireNonNull(timestamp),
                    requireNonNull(timeWindow),
                    requireNonNull(garbageCollectionTime),
                    requireNonNull(garbageCollectionCount));
        }
    }
}
