package com.atlassian.diagnostics.internal.platform.monitor.gc;

import com.atlassian.annotations.nonnull.ReturnValuesAreNonnullByDefault;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.ParametersAreNonnullByDefault;
import java.lang.management.GarbageCollectorMXBean;
import java.time.Clock;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

@ParametersAreNonnullByDefault
@ReturnValuesAreNonnullByDefault
public class GCMXBeanPollerFactory {
    private static final Logger log = LoggerFactory.getLogger(GCMXBeanPollerFactory.class);

    private final Collection<GarbageCollectorMXBean> garbageCollectorMXBeans;
    private final Clock clock;

    public GCMXBeanPollerFactory(final Collection<GarbageCollectorMXBean> garbageCollectorMXBeans,
                                 final Clock clock) {
        this.garbageCollectorMXBeans = garbageCollectorMXBeans;
        this.clock = clock;
    }

    public Optional<GCMXBeanPoller> getGCMXBeansPoller() {
        return getOldGenMXBean().map(garbageCollectorMXBean ->
                new GCMXBeanPoller(garbageCollectorMXBean, clock));
    }

    private Optional<GarbageCollectorMXBean> getOldGenMXBean() {
        final List<GarbageCollectorMXBean> result = garbageCollectorMXBeans.stream()
                .filter(this::isOldGenGCMXBean)
                .collect(Collectors.toList());
        if (result.isEmpty()) {
            log.warn("Received no Old Generation GarbageCollectorMXBean.");
            return Optional.empty();
        }
        if (result.size() > 1) {
            log.warn("Received multiple Old Generation GarbageCollectorMXBeans ({}), using only the first one.", result);
        }
        return Optional.of(result.get(0));
    }

    // Based on
    // https://github.com/elastic/elasticsearch/blob/v7.10.1/server/src/main/java/org/elasticsearch/monitor/jvm/GcNames.java
    private boolean isOldGenGCMXBean(GarbageCollectorMXBean garbageCollectorMXBean) {
        final String gcName = garbageCollectorMXBean.getName();
        return "MarkSweepCompact".equals(gcName) || "PS MarkSweep".equals(gcName)
                || "ConcurrentMarkSweep".equals(gcName) || "G1 Old Generation".equals(gcName);
    }
}
