package com.atlassian.diagnostics.internal.jmx;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.DynamicMBean;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import java.util.concurrent.Callable;

/**
 * A simple implementation of {@link DynamicMBean} that allows you to register a bean with a {@link javax.management.MBeanServer}
 *
 * @since 3.3.0
 */
public class ReadOnlyProxyMBean implements DynamicMBean {

    private final ObjectName sourceObjectName;
    private final MBeanServer mBeanServer;


    public ReadOnlyProxyMBean(final ObjectName sourceObjectName, final MBeanServer mBeanServer) {
        this.sourceObjectName = sourceObjectName;
        this.mBeanServer = mBeanServer;
    }

    @Override
    public Object getAttribute(String attribute) {
        return returnNullOnInstanceNotFoundException(() -> mBeanServer.getAttribute(sourceObjectName, attribute));
    }

    @Override
    public void setAttribute(Attribute attribute) {
        // do nothing
    }

    @Override
    public AttributeList getAttributes(String[] attributes) {
        return returnNullOnInstanceNotFoundException(() -> mBeanServer.getAttributes(sourceObjectName, attributes));
    }

    @Override
    public AttributeList setAttributes(AttributeList attributes) {
        // do nothing
        return new AttributeList();
    }

    @Override
    public Object invoke(String actionName, Object[] params, String[] signature) throws MBeanException, ReflectionException {
        // do nothing
        return null;
    }

    @Override
    public MBeanInfo getMBeanInfo() {
        MBeanAttributeInfo[] sourceObjectAttributes = getAttributesInfo();
        MBeanOperationInfo[] operations = new MBeanOperationInfo[]{};
        return new MBeanInfo(ReadOnlyProxyMBean.class.getName(), "Dynamic MBean", sourceObjectAttributes, null, operations, null);
    }

    private MBeanAttributeInfo[] getAttributesInfo() {
        try {
            return mBeanServer.getMBeanInfo(sourceObjectName).getAttributes();
        } catch (Exception e) {
            return new MBeanAttributeInfo[]{};
        }
    }

    private <T> T returnNullOnInstanceNotFoundException(Callable<T> callable) {
        try {
            return callable.call();
        } catch (InstanceNotFoundException instanceNotFoundException) {
            return null;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

}
