package com.atlassian.diagnostics.internal.ipd.metrics;

import com.atlassian.diagnostics.internal.jmx.ReadOnlyProxyMBean;
import com.atlassian.diagnostics.ipd.api.meters.JmxCopyMeter;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterFactory;
import com.atlassian.diagnostics.ipd.api.registry.IpdRegisterException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.management.DynamicMBean;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import java.util.List;
import java.util.Map;

import static java.lang.management.ManagementFactory.getPlatformMBeanServer;

/**
 * Metric that copies ATTRIBUTES from another JMX bean.
 *
 * @since 3.3.0
 */

public class JmxCopyMeterImpl extends IpdProfilingMeter implements JmxCopyMeter {

    private static final Logger LOG = LoggerFactory.getLogger(JmxCopyMeterImpl.class);

    private final ObjectName sourceBeanName;
    private final DynamicMBean jmxBean;

    protected JmxCopyMeterImpl(final MeterConfig config,
                               final MBeanServer mBeanServer,
                               final ObjectName sourceBeanName,
                               final List<String> allAttributes,
                               final List<String> shortAttributes) {
        super(config, mBeanServer, allAttributes, shortAttributes);
        this.sourceBeanName = sourceBeanName;
        this.jmxBean = new ReadOnlyProxyMBean(sourceBeanName, mBeanServer);
        registerMBean();
    }

    @Override
    public void registerMBean() {
        if (!isEnabled()) {
            return;
        }
        if (mBeanServer.isRegistered(getObjectName())) {
            jmxRegistered.set(true);
        } else {
            synchronized (this) {
                if (!jmxRegistered.get()) {
                    try {
                        mBeanServer.registerMBean(jmxBean, getObjectName());
                        jmxRegistered.set(true);
                    } catch (Exception e) {
                        throw new IpdRegisterException(String.format(
                                "Unable to register JMX bean for metric %s", getMeterKey().getName()), e);
                    }
                }
            }
        }
    }

    private boolean sourceBeanExists() {
        return mBeanServer.isRegistered(sourceBeanName);
    }

    @Override
    public void update() {
        if (!sourceBeanExists()) {
            unregisterMBean();
            removeJmxBeanIfExists();
        } else {
            registerMBean();
            metricUpdated();
        }
    }

    private void removeJmxBeanIfExists() {
        try {
            if (mBeanServer.isRegistered(getObjectName())) {
                mBeanServer.unregisterMBean(getObjectName());
            }
        } catch (Exception e) {
            LOG.warn("Unable to unregister JMX bean for metric {}", getMeterKey().getName(), e);
        }
    }

    @Override
    public Map<String, Object> getAttributes(final boolean extraAttributes) {
        return getAttributes(extraAttributes, sourceBeanName);
    }

    @Override
    public void unregisterMBean() {
        if (!jmxRegistered.compareAndSet(true, false)) {
            return;
        }
        removeJmxBeanIfExists();
    }

    public static MeterFactory<JmxCopyMeter> factory(final ObjectName sourceBeanName,
                                                     final List<String> allAttributes,
                                                     final List<String> shortAttributes) {
        return new MeterFactory<>(config -> create(sourceBeanName, config, allAttributes, shortAttributes), TYPE_ID, "");
    }

    private static JmxCopyMeterImpl create(final ObjectName objectToCopy,
                                           final MeterConfig config,
                                           final List<String> allAttributes,
                                           final List<String> shortAttributes) {
        return new JmxCopyMeterImpl(config, getPlatformMBeanServer(), objectToCopy, allAttributes, shortAttributes);
    }
}
