package com.atlassian.diagnostics.internal.ipd;


import com.atlassian.diagnostics.ipd.api.CachedIpdState;
import com.atlassian.diagnostics.ipd.api.IpdLoggingService;
import com.atlassian.diagnostics.ipd.api.MeterKey;
import com.atlassian.diagnostics.ipd.api.meters.IpdMeter;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.time.Clock;
import java.time.Instant;
import java.util.Map;

import static com.atlassian.diagnostics.ipd.api.IpdConstants.IPD_APP_LOGGER_NAME;
import static com.atlassian.diagnostics.ipd.api.IpdConstants.LOG_LABEL;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * @since 2.2.0
 */
@ParametersAreNonnullByDefault
public class DefaultLoggingService implements IpdLoggingService {
    private final Logger regularLogger = getLogger(IPD_APP_LOGGER_NAME);
    private static final ObjectMapper objectMapper = new ObjectMapper()
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    private final CachedIpdState cachedIpdState;
    private final Clock clock;

    public DefaultLoggingService(CachedIpdState cachedIpdState) {
        this(cachedIpdState, Clock.systemUTC());
    }

    public DefaultLoggingService(CachedIpdState cachedIpdState, Clock clock) {
        this.cachedIpdState = cachedIpdState;
        this.clock = clock;
    }

    @Override
    public void logMetric(final IpdMeter meter) {
        logMetric(meter, clock.instant());
    }

    @Override
    public void logMetric(final IpdMeter meter, final Instant timestamp) {
        if (!meter.isEnabled() || !meter.isVisible()) {
            return;
        }
        final var config = meter.getConfig();
        if (!config.getLoggingCondition().test(meter)) {
            return;
        }
        forceLogMetric(meter, timestamp);
    }

    @Override
    public void forceLogMetric(final IpdMeter meter, final Instant timestamp) {
        final var extraLogging = cachedIpdState.isIpdExtraLoggingEnabled();
        final Map<String, Object> attributes = meter.getAttributes(extraLogging);
        if (attributes.isEmpty()) {
            return;
        }

        final IpdLogEntry logEntry = new IpdLogEntry(String.valueOf(timestamp.getEpochSecond()),
                meter.getMeterKey(),
                extraLogging ? meter.getObjectName().getCanonicalName() : null,
                attributes);
        try {
            final var logger = meter.getConfig().getLogger();
            if (logger.isInfoEnabled()) {
                logger.info(formatData(objectMapper.writeValueAsString(logEntry)));
            }
        } catch (JsonProcessingException e) {
            regularLogger.warn("Can't serialize Jmx instrument: {}", logEntry);
        }
    }

    protected String formatData(final String data) {
        return LOG_LABEL + " " + data;
    }

    static class IpdLogEntry {
        private final String timestamp;
        private final String label;
        private final String objectName;
        private final Map<String, String> tags;
        private final Object attributes;

        public IpdLogEntry(final String timestamp, MeterKey meterKey, @Nullable String objectName, Object attributes) {
            this.timestamp = timestamp;
            this.label = meterKey.getName().toUpperCase();
            this.objectName = objectName;
            this.tags = meterKey.getTagsAsMap().isEmpty() ? null : meterKey.getFormattedTagsAsMap();
            this.attributes = attributes;
        }

        public String getTimestamp() {
            return timestamp;
        }

        public String getLabel() {
            return label;
        }

        public String getObjectName() {
            return objectName;
        }

        public Map<String, String> getTags() {
            return tags;
        }

        public Object getAttributes() {
            return attributes;
        }
    }
}
