package com.atlassian.diagnostics.internal.ipd;

import com.atlassian.diagnostics.ipd.api.CachedIpdState;
import com.atlassian.diagnostics.ipd.api.jobs.IpdJob;
import com.atlassian.diagnostics.ipd.api.jobs.IpdJobRegistration;
import com.atlassian.diagnostics.ipd.api.jobs.IpdJobRegistry;
import com.atlassian.diagnostics.ipd.api.jobs.IpdJobRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Provides default implementation of {@link IpdJobRunner} interface.
 * Product must extend it and implement at least one method {@code isIpdFeatureFlagEnabled();}
 *
 * @since 2.2.0
 */
public class DefaultIpdJobRunner implements IpdJobRunner, IpdJobRegistry {
    private static final Logger log = LoggerFactory.getLogger(DefaultIpdJobRunner.class);

    protected final Map<Class<?>, IpdJobRegistration> jobs;
    private final CachedIpdState cachedIpdState;

    protected DefaultIpdJobRunner(CachedIpdState cachedIpdState) {
        this.cachedIpdState = cachedIpdState;
        this.jobs = new ConcurrentHashMap<>();
    }

    @Override
    public void register(@Nonnull IpdJob job, @Nonnull String key) {
        register(IpdJobRegistration.builder(job, key).build());
    }

    @Override
    public void register(@Nonnull IpdJobRegistration jobRegistration) {
        jobs.put(jobRegistration.getIpdJob().getClass(), jobRegistration);
        log.info("IPD Job {} has been registered", jobRegistration.getIpdJob().getClass().getName());
    }

    @Override
    public void unregister(@Nonnull Class<? extends IpdJob> jobClass) {
        jobs.remove(jobClass);
        log.info("IPD Job {} has been unregistered", jobClass.getName());
    }

    @Override
    public void runJobs() throws InterruptedException {
        if (!cachedIpdState.isIpdEnabled()) {
            log.debug("Not executing IpdJobs because in.product.diagnostics feature flag is disabled");
            return;
        }

        for (IpdJobRegistration registration : jobs.values()) {
            if (Thread.interrupted()) {
                log.info("Thread was interrupted, stopping IPD jobs execution");
                throw new InterruptedException();
            }
            try {
                final var job = registration.getIpdJob();
                if (registration.isWorkInProgress() && !cachedIpdState.isWipIpdEnabled()) {
                    log.debug("Ignoring job {}, in.product.diagnostics.wip feature flag is not enabled", job.getClass().getName());
                    continue;
                }
                log.debug("Running job {}", job.getClass().getName());

                job.runJob();
            } catch (RuntimeException ex) {
                log.error(String.format("Error during executing %s job", registration.getIpdJob().getClass().getName()), ex);
            }
        }
    }

    @Override
    public void shutdown() {
        jobs.clear();
    }
}
