package com.atlassian.diagnostics.internal.analytics;

import com.atlassian.diagnostics.AlertCriteria;
import com.atlassian.diagnostics.CallbackResult;
import com.atlassian.diagnostics.PageRequest;
import com.atlassian.diagnostics.internal.InternalMonitoringService;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.event.events.PluginFrameworkStartedEvent;
import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.JobRunnerRequest;
import com.atlassian.scheduler.JobRunnerResponse;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.config.JobConfig;
import com.atlassian.scheduler.config.JobId;
import com.atlassian.scheduler.config.JobRunnerKey;
import com.atlassian.scheduler.config.RunMode;
import com.atlassian.scheduler.config.Schedule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.PreDestroy;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

public class DailyAlertAnalyticsJob {

    private static final Logger log = LoggerFactory.getLogger(DailyAlertAnalyticsJob.class);

    private static final JobId JOB_ID = JobId.of(DailyAlertAnalyticsJob.class.getName());
    private static final JobRunnerKey JOB_RUNNER_KEY = JobRunnerKey.of(DailyAlertAnalyticsJob.class.getSimpleName());

    private final EventPublisher eventPublisher;
    private final InternalMonitoringService monitoringService;
    private final SchedulerService schedulerService;

    public DailyAlertAnalyticsJob(EventPublisher eventPublisher, InternalMonitoringService monitoringService,
                                  SchedulerService schedulerService) {
        this.eventPublisher = eventPublisher;
        this.monitoringService = monitoringService;
        this.schedulerService = schedulerService;
    }

    @PreDestroy
    public void destroy() {
        schedulerService.unregisterJobRunner(JOB_RUNNER_KEY);
    }

    @EventListener
    public void onFrameworkStarted(PluginFrameworkStartedEvent event) {
        schedulerService.registerJobRunner(JOB_RUNNER_KEY, new AlertAnalyticsJobRunner());
        try {
            schedulerService.scheduleJob(JOB_ID, JobConfig.forJobRunnerKey(JOB_RUNNER_KEY)
                    .withSchedule(Schedule.forCronExpression("0 19 * * * ?")) // collect stats daily at 00:19
                    .withRunMode(RunMode.RUN_ONCE_PER_CLUSTER));
        } catch (SchedulerServiceException e) {
            log.warn("Failed to schedule daily alert analytics publishing job", e);
        }
    }

    private class AlertAnalyticsJobRunner implements JobRunner {

        private static final long MILLIS_IN_DAY = 86400000L;

        @Override
        public JobRunnerResponse runJob(@Nonnull JobRunnerRequest request) {
            long epochDay = Instant.now().minus(1, ChronoUnit.DAYS).toEpochMilli() / MILLIS_IN_DAY;
            Instant since = Instant.ofEpochMilli(epochDay * MILLIS_IN_DAY);
            Instant until = since.plus(1, ChronoUnit.DAYS);

            Map<String, String> nodeHashCache = new HashMap<>();

            monitoringService.internalStreamAlertCounts(AlertCriteria.builder()
                            .since(since)
                            .until(until)
                            .build(),
                    stat -> {
                        Set<String> nodeUuids = stat.getCountsByNodeName().keySet().stream()
                                .map(nodeName -> nodeHashCache.computeIfAbsent(nodeName, AnalyticsUtils::toUuidFormat))
                                .collect(Collectors.toSet());
                        eventPublisher.publish(new DailyAlertCountAnalyticsEvent(
                                epochDay, stat.getIssue(), stat.getPlugin(), nodeUuids, stat.getTotalCount()));
                        return CallbackResult.CONTINUE;
                    },
                    PageRequest.ofSize(250));

            return JobRunnerResponse.success();
        }
    }
}