package com.atlassian.crowd.model;

import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.google.common.base.Function;
import com.google.common.base.Functions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.ImmutableList.copyOf;

/**
 * Utility class for {@link DirectoryEntity}.
 *
 * @since 2.8.1
 */
public class DirectoryEntities {
    private static final Logger log = LoggerFactory.getLogger(DirectoryEntities.class);

    /**
     * A function that projects the name
     */
    public static final Function<DirectoryEntity, String> NAME_FUNCTION = DirectoryEntity::getName;

    /**
     * A function that projects the lower case name
     */
    public static final Function<DirectoryEntity, String> LOWER_NAME_FUNCTION =
            Functions.compose(IdentifierUtils.TO_LOWER_CASE, NAME_FUNCTION);

    /**
     * Transforms entities into their names.
     *
     * @param entities some entities
     * @return their names
     */
    public static Iterable<String> namesOf(Iterable<? extends DirectoryEntity> entities) {
        return Iterables.transform(entities, NAME_FUNCTION);
    }

    /**
     * Remove the duplicate entities from the passed list. If an entity occurs more than once (based on case-insensitive
     * name comparison), all occurrences are excluded from the result.
     *
     * @param remoteEntities the entities to filter.
     * @param <T>            the type of entity to filter.
     * @return the list of entities with duplicates removed.
     */

    public static <T extends DirectoryEntity> List<T> filterOutDuplicates(final List<T> remoteEntities) {
        return filterOutDuplicates(remoteEntities, DirectoryEntity::getName);
    }

    public static <T> List<T> filterOutDuplicates(final Collection<T> remoteEntities, Function<T, String> nameProvider) {
        final Map<String, T> entityMap = Maps.newLinkedHashMap();
        final Set<String> badEntities = Sets.newHashSet();
        for (T remoteEntity : remoteEntities) {
            String remoteName = nameProvider.apply(remoteEntity);
            final String entityId = IdentifierUtils.toLowerCase(remoteName);
            if (!badEntities.contains(entityId)) {
                final T origEntity = entityMap.put(entityId, remoteEntity);
                if (origEntity != null) {
                    entityMap.remove(entityId);
                    badEntities.add(entityId);
                    String origName = nameProvider.apply(origEntity);
                    if (!origName.equals(remoteName)) {
                        log.warn("entity [{}] of type {} duplicated in remote directory by entity [{}]. Ignoring all occurrences.",
                                origName, remoteEntity.getClass().getSimpleName(), remoteName);
                    } else {
                        log.warn("entity [{}] of type {} duplicated in remote directory. Ignoring all occurrences.",
                                origName, remoteEntity.getClass().getSimpleName());
                    }
                }
            }
        }

        return badEntities.isEmpty() && remoteEntities instanceof List ? (List<T>) remoteEntities : copyOf(entityMap.values());
    }
}
