package com.atlassian.crowd.util;

import com.atlassian.crowd.audit.AuditLogContext;
import com.atlassian.crowd.audit.AuditLogEventSource;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.loader.DirectoryInstanceLoader;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationInformation;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationRoundInformation;
import com.atlassian.crowd.event.directory.RemoteDirectorySynchronisationFailedEvent;
import com.atlassian.crowd.event.directory.RemoteDirectorySynchronisationFinishedEvent;
import com.atlassian.crowd.event.directory.RemoteDirectorySynchronisedEvent;
import com.atlassian.crowd.manager.directory.InternalSynchronisationStatusManager;
import com.atlassian.event.api.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;

import static org.apache.commons.lang3.ObjectUtils.firstNonNull;

public class DirectorySynchronisationEventHelper {
    private static final Logger log = LoggerFactory.getLogger(DirectorySynchronisationEventHelper.class);

    public static final long UNKNOWN_TIME_TAKEN_VALUE = -1L;

    private final InternalSynchronisationStatusManager synchronisationStatusManager;
    private final AuditLogContext auditLogContext;
    private final EventPublisher eventPublisher;
    private final DirectoryInstanceLoader directoryInstanceLoader;

    public DirectorySynchronisationEventHelper(final InternalSynchronisationStatusManager synchronisationStatusManager,
                                               final AuditLogContext auditLogContext,
                                               final EventPublisher eventPublisher,
                                               final DirectoryInstanceLoader directoryInstanceLoader) {
        this.synchronisationStatusManager = synchronisationStatusManager;
        this.auditLogContext = auditLogContext;
        this.eventPublisher = eventPublisher;
        this.directoryInstanceLoader = directoryInstanceLoader;
    }

    public void publishDirectorySynchronisationEvent(final Object source, final Directory directory, boolean wasSuccessful, Long customTimeTakenInMs) throws Exception {
        if (wasSuccessful) {
            publishSuccessfulDirectorySynchronisationEvent(source, directory, customTimeTakenInMs);
        } else {
            publishFailedDirectorySynchronisationEvent(source, directory, customTimeTakenInMs);
        }
    }

    public void publishSuccessfulDirectorySynchronisationEvent(final Object source, final Directory directory, Long customTimeTakenInMs) throws Exception {
        final Optional<DirectorySynchronisationRoundInformation> lastRound = getLastRound(directory.getId());
        final RemoteDirectory remoteDirectory = directoryInstanceLoader.getDirectory(directory);
        final long timeTakeInMs = firstNonNull(customTimeTakenInMs, lastRound.map(DirectorySynchronisationRoundInformation::getDurationMs).orElse(null), -1L);
        final RemoteDirectorySynchronisedEvent event = new RemoteDirectorySynchronisedEvent(source, remoteDirectory, lastRound.orElse(null), timeTakeInMs);
        publishSynchronizationEvent(event);
    }

    public void publishFailedDirectorySynchronisationEvent(final Object source, final Directory directory, Long customTimeTakenInMs) throws Exception {
        final Optional<DirectorySynchronisationRoundInformation> lastRound = getLastRound(directory.getId());
        final RemoteDirectory remoteDirectory = directoryInstanceLoader.getDirectory(directory);
        final long timeTakeInMs = firstNonNull(customTimeTakenInMs, lastRound.map(DirectorySynchronisationRoundInformation::getDurationMs).orElse(null), UNKNOWN_TIME_TAKEN_VALUE);
        final RemoteDirectorySynchronisationFailedEvent event = new RemoteDirectorySynchronisationFailedEvent(source, remoteDirectory, lastRound.orElse(null), timeTakeInMs);
        publishSynchronizationEvent(event);
    }

    private void publishSynchronizationEvent(RemoteDirectorySynchronisationFinishedEvent event) throws Exception {
        auditLogContext.withAuditLogSource(AuditLogEventSource.SYNCHRONIZATION, () -> {
            eventPublisher.publish(event);
            return null;
        });
    }

    private Optional<DirectorySynchronisationRoundInformation> getLastRound(final long directoryId) {
        Optional<DirectorySynchronisationRoundInformation> lastRound = Optional.empty();
        try {
            lastRound = Optional.ofNullable(synchronisationStatusManager.getDirectorySynchronisationInformation(directoryId))
                    .map(DirectorySynchronisationInformation::getLastRound);
        } catch (Exception e) {
            log.warn("Could not get last synchronization information for directory {} to create detailed audit log for it", directoryId, e);
        }
        return lastRound;
    }
}
