package com.atlassian.crowd.support;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.manager.application.ApplicationManager;
import com.atlassian.crowd.manager.application.DefaultGroupMembershipService;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;
import com.atlassian.crowd.model.application.RemoteAddress;
import com.atlassian.crowd.service.support.AdditionalSupportInformationService;
import com.google.common.collect.Iterables;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Produces a String containing information about some aspects of the Crowd configuration.
 * This class requires a {@link DirectoryManager}, a {@link DefaultGroupMembershipService}, and optionally, an
 * {@link ApplicationManager} and a {@link AdditionalSupportInformationService}.
 * The latter have been made optional because not all products have multiple applications or provide additional support
 * information.
 *
 * @since 2.6.2
 */
public class SupportInformationServiceImpl implements SupportInformationService {
    private final DirectoryManager directoryManager;
    private final DefaultGroupMembershipService defaultGroupMembershipService;

    @Nullable
    private ApplicationManager applicationManager;

    private List<AdditionalSupportInformationService> additionalSupportInformationServices = Collections.emptyList();

    private static final String ATTRIBUTES_KEY = "Attributes";
    private static final Logger log = LoggerFactory.getLogger(SupportInformationServiceImpl.class);

    public SupportInformationServiceImpl(final DirectoryManager directoryManager,
                                         final DefaultGroupMembershipService defaultGroupMembershipService) {
        this.directoryManager = checkNotNull(directoryManager);
        this.defaultGroupMembershipService = defaultGroupMembershipService;
    }

    public String getSupportInformation(@Nullable User currentUser) {
        final Map<String, String> map = getSupportInformationMap(currentUser);
        StringBuilder builder = new StringBuilder(1000);
        List<String> previousCategories = Collections.emptyList();

        for (Map.Entry<String, String> entry : map.entrySet()) {
            List<String> parts = Arrays.asList(entry.getKey().split("\\."));

            // attributes contain package names with dots, merge them back
            final int splitIndex = parts.indexOf(ATTRIBUTES_KEY) + 1;
            if (splitIndex > 0 && splitIndex < parts.size() - 1) { // do not split if ATTRIBUTES_KEY is at the end of the string
                parts = Stream.concat(
                        parts.subList(0, splitIndex).stream(),
                        Stream.of(String.join(".", parts.subList(splitIndex, parts.size())))
                ).collect(Collectors.toList());
            }

            List<String> categories = parts.subList(0, parts.size() - 1);
            String attribute = parts.get(parts.size() - 1);

            boolean parentWasDifferent = false;
            for (int i = 0; i < categories.size(); i++) {
                if (parentWasDifferent || i >= previousCategories.size() || !categories.get(i).equals(previousCategories.get(i))) {
                    builder.append(StringUtils.repeat("\t", i)).append(categories.get(i)).append(":\n");
                    parentWasDifferent = true;
                }
            }

            builder.append(StringUtils.repeat("\t", categories.size())).append(attribute).append(": ")
                    .append(entry.getValue()).append("\n");

            previousCategories = categories;
        }
        return builder.toString();
    }

    public Map<String, String> getSupportInformationMap(@Nullable User currentUser) {
        SupportInformationBuilder builder = new SupportInformationBuilder();

        addCurrentUserInformation(builder, currentUser);
        addDirectoryConfiguration(builder);
        addApplicationConfiguration(builder);
        addAdditionalSupportInformation(builder);

        return builder.getMap();
    }

    private void addAdditionalSupportInformation(SupportInformationBuilder builder) {
        for (AdditionalSupportInformationService additionalSupportInformationService : additionalSupportInformationServices) {
            additionalSupportInformationService.extendSupportInformation(builder);
        }
    }

    private void addCurrentUserInformation(SupportInformationBuilder builder, @Nullable User currentUser) {
        if (currentUser != null) {
            builder.prefix("Current user")
                    .field("Directory ID", currentUser.getDirectoryId())
                    .field("Username", currentUser.getName())
                    .field("Display name", currentUser.getDisplayName())
                    .field("Email address", currentUser.getEmailAddress());
        }
    }

    private void addDirectoryConfiguration(SupportInformationBuilder builder) {
        List<Directory> directories = directoryManager.findAllDirectories();
        if (directories != null) {
            int index = 1;
            for (Directory directory : directories) {
                builder.prefix("Directory " + index)
                        .field("Directory ID", directory.getId())
                        .field("Name", directory.getName())
                        .field("Active", directory.isActive())
                        .field("Type", directory.getType())
                        .field("Created date", directory.getCreatedDate())
                        .field("Updated date", directory.getUpdatedDate())
                        .field("Allowed operations", directory.getAllowedOperations())
                        .field("Implementation class", directory.getImplementationClass())
                        .field("Encryption type", directory.getEncryptionType())
                        .attributes(ATTRIBUTES_KEY, directory.getAttributes());
                index++;
            }
        }
    }

    private void addApplicationConfiguration(SupportInformationBuilder builder) {
        if (applicationManager != null) {
            AtomicInteger applicationIndex = new AtomicInteger(1);
            for (Application application : applicationManager.findAll()) {
                builder.prefix("Application " + applicationIndex.get())
                        .field("Application ID", application.getId())
                        .field("Name", application.getName())
                        .field("Active", application.isActive())
                        .field("Type", application.getType())
                        .field("Description", application.getDescription())
                        .field("Is lowercase output", application.isLowerCaseOutput())
                        .field("Is aliasing enabled", application.isAliasingEnabled())
                        .field("Remote addresses",
                                Iterables.transform(application.getRemoteAddresses(), RemoteAddress::getAddress))
                        .field("Created date", application.getCreatedDate())
                        .field("Updated date", application.getUpdatedDate())
                        .attributes(ATTRIBUTES_KEY, application.getAttributes());
                int directoryIndex = 1;
                for (ApplicationDirectoryMapping directoryMapping : application.getApplicationDirectoryMappings()) {
                    builder.prefix("Application " + applicationIndex.get() + ".Mapping " + directoryIndex)
                            .field("Mapped to directory ID", directoryMapping.getDirectory().getId())
                            .field("Allow all to authenticate", directoryMapping.isAllowAllToAuthenticate())
                            .field("Mapped groups", directoryMapping.getAuthorisedGroupNames())
                            .field("Allowed operations", directoryMapping.getAllowedOperations());
                    final List<String> defaultGroupMemberships = fetchDefaultGroupMemberships(application, directoryMapping);
                    if (!defaultGroupMemberships.isEmpty()) {
                        builder.field("Default group memberships", defaultGroupMemberships);
                    }
                    directoryIndex++;
                }
                additionalSupportInformationServices.forEach(s -> s.extendSupportInformation(builder, application, applicationIndex.get()));
                applicationIndex.incrementAndGet();
            }
        }
    }

    private List<String> fetchDefaultGroupMemberships(Application application, ApplicationDirectoryMapping directoryMapping) {
        try {
            return defaultGroupMembershipService.listAll(application, directoryMapping);
        } catch (OperationFailedException e) {
            log.debug("Could not fetch default group memberships for application {} and directory {}",
                    application.getId(),
                    directoryMapping.getDirectory().getId(),
                    e);
            return Collections.emptyList();
        }
    }

    public void setApplicationManager(ApplicationManager applicationManager) {
        this.applicationManager = applicationManager;
    }

    public void setAdditionalSupportInformationServices(@Nullable List<AdditionalSupportInformationService> additionalSupportInformationServices) {
        this.additionalSupportInformationServices = additionalSupportInformationServices;
    }
}
