package com.atlassian.crowd.manager.recovery;

import com.atlassian.crowd.embedded.api.ApplicationFactory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.event.EventStore;
import com.atlassian.crowd.manager.application.ApplicationServiceGeneric;
import com.atlassian.crowd.manager.application.AuthenticationOrderOptimizer;
import com.atlassian.crowd.manager.application.search.SearchStrategyFactory;
import com.atlassian.crowd.manager.avatar.AvatarProvider;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.manager.permission.PermissionManager;
import com.atlassian.crowd.manager.webhook.WebhookRegistry;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.user.User;
import com.atlassian.event.api.EventPublisher;
import com.google.common.collect.ImmutableList;

import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * An extension of {@link ApplicationServiceGeneric} that supports recovery mode. When recovery mode is activated,
 * an extra directory as provided by the {@link RecoveryModeService#getRecoveryDirectory() RecoveryModeService} will
 * be prepended to the list of directories for any application.
 *
 * @since 2.7.2
 */
public class RecoveryModeAwareApplicationService extends ApplicationServiceGeneric {
    private final RecoveryModeService recoveryModeService;
    private final ApplicationFactory applicationFactory;

    public RecoveryModeAwareApplicationService(DirectoryManager directoryManager,
                                               SearchStrategyFactory searchStrategyFactory,
                                               PermissionManager permissionManager,
                                               EventPublisher eventPublisher,
                                               EventStore eventStore,
                                               WebhookRegistry webhookRegistry,
                                               AvatarProvider avatarProvider,
                                               ApplicationFactory applicationFactory,
                                               RecoveryModeService recoveryModeService,
                                               AuthenticationOrderOptimizer authenticationOrderOptimizer) {
        super(directoryManager, searchStrategyFactory, permissionManager, eventPublisher, eventStore, webhookRegistry, avatarProvider, authenticationOrderOptimizer);
        this.applicationFactory = checkNotNull(applicationFactory);
        this.recoveryModeService = checkNotNull(recoveryModeService, "recoveryModeService");
    }

    @Override
    protected List<Directory> getActiveDirectories(Application application) {
        List<Directory> directories = super.getActiveDirectories(application);
        if (supportsRecoveryLogin(application)) {
            // add extra 'recovery mode' directory
            return ImmutableList.<Directory>builder()
                    .add(recoveryModeService.getRecoveryDirectory())
                    .addAll(directories)
                    .build();
        } else {
            return directories;
        }
    }

    @Override
    public boolean isUserAuthorised(Application application, String username) {
        return isRecoveryUserAuthorisation(application, username) || super.isUserAuthorised(application, username);
    }

    @Override
    public boolean isUserAuthorised(final Application application, final User user) {
        return isRecoveryUserAuthorisation(application, user.getName()) || super.isUserAuthorised(application, user);
    }

    private boolean isRecoveryUserAuthorisation(final Application application, final String username) {
        return supportsRecoveryLogin(application)
                && IdentifierUtils.equalsInLowerCase(username, recoveryModeService.getRecoveryUsername());
    }

    private boolean supportsRecoveryLogin(final Application application) {
        return recoveryModeService.isRecoveryModeOn()
                && application.getId().equals(applicationFactory.getApplication().getId());
    }
}
