package com.atlassian.crowd.manager.permission;

import com.atlassian.crowd.dao.application.ApplicationDAO;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.OperationType;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.event.application.ApplicationUpdatedEvent;
import com.atlassian.crowd.event.directory.DirectoryUpdatedEvent;
import com.atlassian.crowd.exception.ApplicationNotFoundException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;
import com.atlassian.crowd.model.application.DirectoryMapping;
import com.atlassian.crowd.model.application.ImmutableApplication;
import com.atlassian.crowd.model.directory.ImmutableDirectory;
import com.atlassian.event.api.EventPublisher;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Implementation of the {@link PermissionManager}.
 */
@Transactional
public class PermissionManagerImpl implements PermissionManager {
    private static final Logger logger = LoggerFactory.getLogger(PermissionManagerImpl.class);

    private final ApplicationDAO applicationDao;
    private final DirectoryDao directoryDao;
    private final EventPublisher eventPublisher;

    public PermissionManagerImpl(ApplicationDAO applicationDao, DirectoryDao directoryDao, EventPublisher eventPublisher) {
        this.applicationDao = checkNotNull(applicationDao);
        this.directoryDao = checkNotNull(directoryDao);
        this.eventPublisher = eventPublisher;
    }

    public boolean hasPermission(final Directory directory, final OperationType operationType) {
        Validate.notNull(directory, "directory cannot be null");
        Validate.notNull(operationType, "operationType cannot be null");

        boolean permission = directory.getAllowedOperations().contains(operationType);

        if (!permission && logger.isDebugEnabled()) {
            logger.debug("Directory " + directory.getName() + " : Permission " + operationType.name() + " has been denied");
        }

        return permission;
    }

    public boolean hasPermission(final Application application, final Directory directory, final OperationType operationType) {
        Validate.notNull(application, "application cannot be null");
        Validate.notNull(directory, "directory cannot be null");
        Validate.notNull(operationType, "operationType cannot be null");

        boolean hasPermission = false;

        if (hasPermission(directory, operationType)) {
            ApplicationDirectoryMapping mapping = application.getApplicationDirectoryMapping(directory.getId());
            if (mapping != null) {
                hasPermission = mapping.getAllowedOperations().contains(operationType);
            }
        }

        return hasPermission;
    }

    public void removePermission(final Directory directory, final OperationType operationType)
            throws DirectoryNotFoundException {
        Validate.notNull(directory, "directory cannot be null");
        Validate.notNull(operationType, "operationType cannot be null");

        final Directory oldDirectory = ImmutableDirectory.from(directoryDao.findById(directory.getId()));
        directory.getAllowedOperations().remove(operationType);
        directoryDao.update(directory);
        final Directory newDirectory = ImmutableDirectory.from(directoryDao.findById(directory.getId()));
        eventPublisher.publish(new DirectoryUpdatedEvent(this, oldDirectory, newDirectory));
    }

    public void removePermission(final Application application, final Directory directory, final OperationType operationType)
            throws ApplicationNotFoundException {
        Validate.notNull(application, "application cannot be null");
        Validate.notNull(directory, "directory cannot be null");
        Validate.notNull(operationType, "operationType cannot be null");

        ApplicationDirectoryMapping mapping = application.getApplicationDirectoryMapping(directory.getId());
        if (mapping != null) {
            final Application oldApplication = ImmutableApplication.from(applicationDao.findById(application.getId()));
            mapping.getAllowedOperations().remove(operationType);
            applicationDao.update(application);
            final Application newApplication = ImmutableApplication.from(applicationDao.findById(application.getId()));
            eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, newApplication));
        }
    }

    public void addPermission(final Directory directory, final OperationType operationType)
            throws DirectoryNotFoundException {
        Validate.notNull(directory, "directory cannot be null");
        Validate.notNull(operationType, "operationType cannot be null");

        final Directory oldDirectory = ImmutableDirectory.from(directoryDao.findById(directory.getId()));
        directory.getAllowedOperations().add(operationType);
        directoryDao.update(directory);
        final Directory newDirectory = ImmutableDirectory.from(directoryDao.findById(directory.getId()));
        eventPublisher.publish(new DirectoryUpdatedEvent(this, oldDirectory, newDirectory));
    }

    public void addPermission(final Application application, final Directory directory, final OperationType operationType)
            throws ApplicationNotFoundException {
        Validate.notNull(application, "application cannot be null");
        Validate.notNull(directory, "directory cannot be null");
        Validate.notNull(operationType, "operationType cannot be null");

        DirectoryMapping mapping = application.getDirectoryMapping(directory.getId());
        if (mapping != null) {
            final Application oldApplication = ImmutableApplication.from(applicationDao.findById(application.getId()));
            mapping.getAllowedOperations().add(operationType);
            applicationDao.update(application);
            final Application newApplication = ImmutableApplication.from(applicationDao.findById(application.getId()));
            eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, newApplication));
        }
    }
}
