package com.atlassian.crowd.manager.directory.nestedgroups;

import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupType;
import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;

import java.util.Collection;
import java.util.function.Function;

/**
 * Builder class for {@link NestedGroupsProvider}
 */
public class NestedGroupsProviderBuilder {
    private MultipleGroupsProvider provider;
    private Function<String, String> idNormalizer;
    private Function<Group, String> idProvider;
    private int batchSize;

    private NestedGroupsProviderBuilder() {
    }

    public static NestedGroupsProviderBuilder create() {
        return new NestedGroupsProviderBuilder();
    }

    public NestedGroupsProviderBuilder setProvider(MultipleGroupsProvider provider) {
        this.provider = provider;
        return this;
    }

    public NestedGroupsProviderBuilder setSingleGroupProvider(SingleGroupProvider provider) {
        setProvider((names) -> {
            ListMultimap<String, Group> results = ArrayListMultimap.create();
            for (String name : names) {
                results.putAll(name, provider.getDirectlyRelatedGroups(name));
            }
            return results;
        });
        setBatchSize(1);
        return this;
    }

    /**
     * Instructs to use case-insensitive group names as identifiers.
     */
    public NestedGroupsProviderBuilder useGroupName() {
        return setIdProvider(Group::getName).setIdNormalizer(IdentifierUtils::toLowerCase);
    }

    /**
     * Instructs to use case-sensitive external ids as identifiers.
     */
    public NestedGroupsProviderBuilder useExternalId() {
        return setIdProvider(Group::getExternalId).setIdNormalizer(Function.identity());
    }

    /**
     * Sets the batch size for membership query.
     * @param batchSize maximum number of groups passed to {@link NestedGroupsProvider#getDirectlyRelatedGroups(Collection)}
     */
    public NestedGroupsProviderBuilder setBatchSize(int batchSize) {
        this.batchSize = batchSize;
        return this;
    }

    public NestedGroupsProvider build() {
        return new NestedGroupsProviderImpl(provider, idNormalizer, idProvider, batchSize);
    }

    private NestedGroupsProviderBuilder setIdNormalizer(Function<String, String> idNormalizer) {
        this.idNormalizer = idNormalizer;
        return this;
    }

    private NestedGroupsProviderBuilder setIdProvider(Function<Group, String> idProvider) {
        this.idProvider = idProvider;
        return this;
    }

    public void useCache(NestedGroupsCacheProvider cacheProvider, long directoryId, boolean isChildrenQuery, GroupType groupType) {
        Preconditions.checkNotNull(idNormalizer, "All fields should be set before calling useCache");
        Preconditions.checkNotNull(provider, "All fields should be set before calling useCache");
        provider = new CachedMultipleGroupsProvider(
                cacheProvider.getSubgroupsCache(directoryId, isChildrenQuery, groupType),
                cacheProvider.getGroupsCache(directoryId, groupType),
                idNormalizer, provider);
    }
}
