package com.atlassian.crowd.manager.directory.nestedgroups;

import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.group.Group;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.function.Function;

/**
 * Class simplifying iterating over nested groups.
 * @param <T> either {@link Group} or {@link String}
 */
public class NestedGroupsIterator<T> {
    final private Queue<String> groupsToVisit = new LinkedList<>();
    final private Set<String> processed = new HashSet<>();
    final private Queue<T> toReturn;
    final private NestedGroupsProvider provider;
    final private Function<Group, T> toReturnTransformer;

    private NestedGroupsIterator(Collection<String> toVisit,
              Collection<? extends T> initialToReturn,
              NestedGroupsProvider provider,
              final Function<Group, T> toReturnTransformer) {
        this.groupsToVisit.addAll(toVisit);
        this.processed.addAll(Collections2.transform(toVisit, provider::normalizeIdentifier));
        this.toReturn = new LinkedList<>(initialToReturn);
        this.provider = provider;
        this.toReturnTransformer = toReturnTransformer;
    }

    private void fetchRelatedGroups(List<String> groupNames) throws OperationFailedException {
        for (Group member : provider.getDirectlyRelatedGroups(groupNames)) {
            String name = provider.getIdentifierForSubGroupsQuery(member);
            if (processed.add(provider.normalizeIdentifier(name))) {
                groupsToVisit.add(name);
                toReturn.add(toReturnTransformer.apply(member));
            }
        }
    }

    public boolean hasNext() throws OperationFailedException {
        fetchNextIfNeeded();
        return !toReturn.isEmpty();
    }

    public T next() throws OperationFailedException {
        fetchNextIfNeeded();
        return toReturn.remove();
    }

    private void fetchNextIfNeeded() throws OperationFailedException {
        while (toReturn.isEmpty() && !groupsToVisit.isEmpty()) {
            List<String> batch = new ArrayList<>();
            while (batch.size() < provider.getMaxBatchSize() && !groupsToVisit.isEmpty()) {
                batch.add(groupsToVisit.remove());
            }
            fetchRelatedGroups(batch);
        }
    }

    public List<T> toList() throws OperationFailedException {
        return nextBatch(Integer.MAX_VALUE);
    }

    public List<T> nextBatch(int maxBatchSize) throws OperationFailedException {
        List<T> batch = new ArrayList<>();
        while (batch.size() < maxBatchSize && hasNext()) {
            batch.add(next());
        }
        return batch;
    }

    public boolean anyMatch(PredicateWithException<T> consumer) throws OperationFailedException {
        while (hasNext()) {
            if (consumer.test(next())) {
                return true;
            }
        }
        return false;
    }

    public void visitAll() throws OperationFailedException {
        while (hasNext()) {
            next();
        }
    }

    public static NestedGroupsIterator<Group> groupsIterator(
            Collection<? extends Group> groups, boolean includeOriginal, NestedGroupsProvider provider) {
        return new NestedGroupsIterator<>(
                Collections2.transform(groups, Group::getName),
                includeOriginal ? groups : ImmutableList.of(),
                provider,
                Function.identity());
    }

    public static NestedGroupsIterator<String> namesIterator(
            Collection<String> groups, boolean includeOriginal, NestedGroupsProvider provider) {
        return new NestedGroupsIterator<>(
                groups, includeOriginal ? groups : ImmutableList.of(), provider, Group::getName);
    }

    public static NestedGroupsIterator<Group> groupsIterator(
            Collection<String> groups, NestedGroupsProvider provider) {
        return new NestedGroupsIterator<>(groups, ImmutableList.of(), provider, Function.identity());
    }

    public interface PredicateWithException<T> {
        boolean test(T t) throws OperationFailedException;
    }
}
