package com.atlassian.crowd.manager.directory.nestedgroups;

import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupType;
import com.google.common.base.Throwables;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import java.util.Arrays;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * Cache provider for {@link CachedMultipleGroupsProvider}
 */
public class NestedGroupsCacheProvider {
    private static final long GROUPS_CACHE_MARGIN_MS = 2000;
    private static final String SUBGROUPS_CACHE_EXPIRY_MS_PROPERTY = "crowd.manager.directory.searcher.subgroups.cache.expiry.ms";
    private static final String SUBGROUPS_CACHE_MAX_SIZE_PROPERTY = "crowd.manager.directory.searcher.subgroups.cache.maxSize";


    private final long expiryMs;
    private final int maxSize;
    private final Cache<Object, Cache<String, String[]>> subgroupCaches;
    // Purpose of these caches is to save memory by de-duplicating groups from the same directory. Their expiration
    // times are extended by GROUPS_CACHE_MARGIN_MS, to make sure that the entries pointed by values of subgroupCaches
    // are not expired.
    private final Cache<Object, Cache<String, Group>> groupCaches;

    public NestedGroupsCacheProvider(long expiryMs, int maxSize) {
        this.expiryMs = expiryMs;
        this.maxSize = maxSize;
        this.subgroupCaches = createCache(expiryMs, false);
        this.groupCaches = createCache(expiryMs + GROUPS_CACHE_MARGIN_MS, false);
    }

    private Cache<String, String[]> createSubgroupsCache() {
        return createCache(expiryMs, true);
    }

    private Cache<String, Group> createGroupsCache() {
        return createCache(expiryMs + GROUPS_CACHE_MARGIN_MS, true);
    }

    private <F, T> Cache<F, T> createCache(long expiryMs, boolean expireAfterWrite) {
        CacheBuilder builder = CacheBuilder.newBuilder().maximumSize(maxSize);
        return expireAfterWrite ? builder.expireAfterWrite(expiryMs, TimeUnit.MILLISECONDS).build()
                : builder.expireAfterAccess(expiryMs, TimeUnit.MILLISECONDS).build();
    }

    /**
     * Returns cache for a given set of parameters. The same cache instance will be returned for the same parameters,
     * if subsequent calls do not exceed configured expiration time.
     * Cache entries of the returned cache are expired after write with configured expiry time.
     */
    protected Cache<String, String[]> getSubgroupsCache(long directoryId, boolean isChildrenQuery, GroupType type) {
        try {
            return subgroupCaches.get(Arrays.asList(directoryId, isChildrenQuery, type), this::createSubgroupsCache);
        } catch (ExecutionException e) {
            Throwables.propagateIfPossible(e.getCause());
            throw new RuntimeException(e.getCause());
        }
    }

    /**
     * Returns cache for a given set of parameters. The same cache instance will be returned for the same parameters,
     * if subsequent calls do not exceed configured expiration time plus margin described below.
     * Cache entries of the returned cache are expired after write with configured expiry time plus margin.
     * Comparing to {@link #getSubgroupsCache(long, boolean, GroupType)} expiration times are extended by
     * {@link #GROUPS_CACHE_MARGIN_MS} to make sure that values that are pointed at are not expired.
     */
    protected Cache<String, Group> getGroupsCache(long directoryId, GroupType type) {
        try {
            return groupCaches.get(Arrays.asList(directoryId, type), this::createGroupsCache);
        } catch (ExecutionException e) {
            Throwables.propagateIfPossible(e.getCause());
            throw new RuntimeException(e.getCause());
        }
    }

    /**
     * Returns provider configured according to system properties or {@link Optional#empty()}, if not configured.
     * Please note that the cache is experimental and that's why it's disabled by default.
     */
    public static Optional<NestedGroupsCacheProvider> createFromSystemProperties() {
        final long expiry = Long.getLong(SUBGROUPS_CACHE_EXPIRY_MS_PROPERTY, 0);
        final int size = Integer.getInteger(SUBGROUPS_CACHE_MAX_SIZE_PROPERTY, 0);
        return (expiry > 0 && size > 0) ? Optional.of(new NestedGroupsCacheProvider(expiry, size)) : Optional.empty();
    }
}
