package com.atlassian.crowd.manager.directory;

import com.atlassian.crowd.directory.SynchronisableDirectoryProperties;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationInformation;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationRoundInformation;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.event.migration.XMLRestoreStartedEvent;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.model.directory.DirectorySynchronisationStatus;
import com.atlassian.crowd.model.directory.SynchronisationStatusKey;
import com.atlassian.crowd.service.cluster.ClusterNode;
import com.atlassian.crowd.service.cluster.ClusterService;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import java.io.Serializable;
import java.time.Clock;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

public class SynchronisationStatusManagerImpl implements InternalSynchronisationStatusManager {
    private static final Logger logger = LoggerFactory.getLogger(SynchronisationStatusManagerImpl.class);

    private final DirectorySynchronisationInformationStore store;
    private final DirectoryDao directoryDao;
    private final Clock clock;
    private final DirectorySynchronisationTokenStore directorySynchronisationTokenStore;
    private final ClusterService clusterService;

    public SynchronisationStatusManagerImpl(DirectorySynchronisationInformationStore store, EventPublisher eventPublisher,
                                            DirectoryDao directoryDao,
                                            Clock clock,
                                            DirectorySynchronisationTokenStore directorySynchronisationTokenStore,
                                            ClusterService clusterService) {
        this.store = store;
        this.directoryDao = directoryDao;
        this.clock = clock;
        this.directorySynchronisationTokenStore = directorySynchronisationTokenStore;
        this.clusterService = clusterService;
        eventPublisher.register(this);
    }

    @Override
    public void syncStarted(Directory directory) {
        store.syncStarted(directory.getId(), clock.millis());
    }


    @Override
    public void syncStatus(long directoryId, String key, Serializable... parameters) {
        try {
            final Optional<SynchronisationStatusKey> maybeEnum = SynchronisationStatusKey.fromKey(key);
            if (maybeEnum.isPresent()) {
                store.syncStatus(directoryId, maybeEnum.get(), Arrays.asList(parameters));
            } else {
                store.syncStatus(directoryId, key, Arrays.asList(parameters));
            }
        } catch (Exception e) {
            logger.warn("Could not update synchronisation status for directory {}, status {}, status parameters {}", directoryId, key, parameters);
            logger.warn("Caused by: ", e);
        }
    }

    @Override
    public void syncStatus(long directoryId, SynchronisationStatusKey statusKey, List<Serializable> parameters) {
        try {
            store.syncStatus(directoryId, statusKey, parameters);
        } catch (Exception e) {
            logger.warn("Could not update synchronisation status for directory {}, status {}, status parameters {}", directoryId, statusKey, parameters);
            logger.warn("Caused by: ", e);
        }
    }

    @Override
    public void syncFinished(long directoryId) {
        final DirectorySynchronisationRoundInformation active = store.getActive(directoryId);
        if (active != null) {
            final SynchronisationStatusKey key = SynchronisationStatusKey.fromKey(active.getStatusKey())
                    .orElseThrow(() -> new IllegalStateException("Can't finish synchronisation status"));
            store.syncFinished(directoryId, clock.millis(), key, active.getStatusParameters());
        }
    }

    @Override
    public void syncFailure(long directoryId, SynchronisationMode syncMode, Throwable throwable) {
        store.syncFailure(directoryId, syncMode, ExceptionUtils.getMessage(throwable));
    }

    @Override
    public void syncFinished(long directoryId, SynchronisationStatusKey statusKey, List<Serializable> parameters) {
        store.syncFinished(directoryId, clock.millis(), statusKey, parameters);
    }

    @Override
    public DirectorySynchronisationInformation getDirectorySynchronisationInformation(Directory directory) {
        final DirectorySynchronisationInformation info = new DirectorySynchronisationInformation(store.getLast(directory.getId()).orElse(null),
                store.getActive(directory.getId()));
        if (info.getActiveRound() == null && info.getLastRound() == null) {
            final long startTime = NumberUtils.toLong(directory.getValue(SynchronisableDirectoryProperties.LAST_START_SYNC_TIME), 0);
            final long duration = NumberUtils.toLong(directory.getValue(SynchronisableDirectoryProperties.LAST_SYNC_DURATION_MS), 0);
            final Optional<ClusterNode> node = clusterService.getClusterNode();

            final DirectorySynchronisationRoundInformation lastRound = startTime == 0 ? null : DirectorySynchronisationRoundInformation.builder()
                    .setStartTime(startTime)
                    .setDurationMs(duration)
                    .setNodeId(node.map(ClusterNode::getNodeId).orElse(null))
                    .setNodeName(node.map(ClusterNode::getNodeName).orElse(null))
                    .build();

            return new DirectorySynchronisationInformation(lastRound, null);
        } else if (info.getActiveRound() != null) {
            final DirectorySynchronisationRoundInformation activeRound = DirectorySynchronisationRoundInformation.builder(info.getActiveRound())
                    .setDurationMs(clock.millis() - info.getActiveRound().getStartTime())
                    .build();
            return new DirectorySynchronisationInformation(info.getLastRound(), activeRound);
        } else {
            return info;
        }
    }

    @Override
    @Transactional
    public DirectorySynchronisationInformation getDirectorySynchronisationInformation(long directoryId) throws DirectoryNotFoundException {
        return getDirectorySynchronisationInformation(directoryDao.findById(directoryId));
    }

    @Override
    @Transactional
    public String getLastSynchronisationTokenForDirectory(long directoryId) {
        return directorySynchronisationTokenStore.getLastSynchronisationTokenForDirectory(directoryId);
    }

    @Override
    @Transactional
    public void storeSynchronisationTokenForDirectory(long directoryId, String synchronisationToken) {
        directorySynchronisationTokenStore.storeSynchronisationTokenForDirectory(directoryId, synchronisationToken);
    }

    @Override
    public void removeStatusesForDirectory(long directoryId) {
        store.clear(directoryId);
    }

    @Override
    public Collection<DirectorySynchronisationStatus> getStalledSynchronizations() {
        return store.getStalledSynchronizations();
    }

    @Override
    @Transactional
    public void clearSynchronisationTokenForDirectory(long directoryId) {
        directorySynchronisationTokenStore.clearSynchronisationTokenForDirectory(directoryId);
    }

    @EventListener
    public void handleEvent(XMLRestoreStartedEvent event) {
        store.clear();
    }
}
