package com.atlassian.crowd.manager.directory;

import com.atlassian.beehive.ClusterLock;
import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationRoundInformation;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerManager;
import com.atlassian.crowd.model.directory.DirectorySynchronisationStatus;
import com.atlassian.crowd.model.directory.SynchronisationStatusKey;
import com.atlassian.crowd.util.DirectorySynchronisationEventHelper;
import org.slf4j.Logger;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import static com.atlassian.crowd.util.DirectorySynchronisationEventHelper.UNKNOWN_TIME_TAKEN_VALUE;
import static org.slf4j.LoggerFactory.getLogger;

@Transactional
public class FailedSynchronisationManagerImpl implements FailedSynchronisationManager {
    private static final Logger log = getLogger(FailedSynchronisationManagerImpl.class);

    private final InternalSynchronisationStatusManager synchronisationStatusManager;
    private final DirectoryPollerManager pollerManager;
    private final ClusterLockService lockService;
    private final DirectoryManager directoryManager;
    private final DirectorySynchronisationEventHelper syncEventHelper;

    public FailedSynchronisationManagerImpl(InternalSynchronisationStatusManager synchronisationStatusManager,
                                            DirectoryPollerManager pollerManager, ClusterLockService lockService,
                                            DirectoryManager directoryManager, DirectorySynchronisationEventHelper syncEventHelper) {
        this.synchronisationStatusManager = synchronisationStatusManager;
        this.pollerManager = pollerManager;
        this.lockService = lockService;
        this.directoryManager = directoryManager;
        this.syncEventHelper = syncEventHelper;
    }

    @Override
    public void finalizeSynchronisationStatuses() {
        directoryManager.findAllDirectories().forEach(directory -> {
            final ClusterLock directoryLock = lockService.getLockForName(DirectorySynchronisationUtils.getLockName(directory.getId()));
            if (directoryLock.tryLock()) {
                try {
                    final DirectorySynchronisationRoundInformation activeStatus = synchronisationStatusManager.getDirectorySynchronisationInformation(directory.getId()).getActiveRound();
                    if (activeStatus != null) {
                        finalizeSynchronisationStatusAndPublishAuditEvent(directory);
                    }
                } catch (DirectoryNotFoundException e) {
                    log.warn("Couldn't check synchronisation status for directory {}", directory.getId(), e);
                } catch (Exception e) {
                    // Nothing really bad happened, try to clean next directory status if necessary
                    log.warn("Couldn't finish synchronisation status for directory {}", directory.getId(), e);
                } finally {
                    directoryLock.unlock();
                }
            } else {
                log.debug("Not checking directory {}, lock unavailable", directory.getId());
            }
        });
    }

    private void finalizeSynchronisationStatusAndPublishAuditEvent(Directory directory) {
        log.info("Found not final synchronisation status for directory {}", directory.getId());
        setSynchronisationStatusAndPublishSynchronisationFailedEvent(directory, SynchronisationStatusKey.ABORTED);
        synchronisationStatusManager.clearSynchronisationTokenForDirectory(directory.getId());
        log.info("Fixed stale synchronisation status for directory {}", directory.getId());
    }

    @Override
    public int rescheduleStalledSynchronisations() {
        final Collection<Directory> stalledSyncsDirs = synchronisationStatusManager.getStalledSynchronizations().stream().map(DirectorySynchronisationStatus::getDirectory).collect(Collectors.toList());
        final AtomicInteger rescheduledSyncs = new AtomicInteger(0);
        if (stalledSyncsDirs.size() > 0) {
            log.info("Found {} stalled synchronisations for directories [ {} ]. Rescheduling them to run again", stalledSyncsDirs.size(), stalledSyncsDirs.stream().map(Directory::getId).collect(Collectors.toList()));
            stalledSyncsDirs.forEach(directory -> {
                final ClusterLock lock = lockService.getLockForName(DirectorySynchronisationUtils.getLockName(directory.getId()));
                if (lock.tryLock()) {
                    try {
                        setSynchronisationStatusAndPublishSynchronisationFailedEvent(directory, SynchronisationStatusKey.FAILURE);
                        pollerManager.triggerPoll(directory.getId(), SynchronisationMode.FULL);
                        rescheduledSyncs.incrementAndGet();
                    } finally {
                        lock.unlock();
                    }
                } else {
                    log.debug("Couldn't acquire cluster lock for directory {} - ignoring", directory);
                }
            });
        } else {
            log.debug("Didn't find any stalled synchronisation");
        }
        return rescheduledSyncs.get();
    }

    private void setSynchronisationStatusAndPublishSynchronisationFailedEvent(Directory directory, SynchronisationStatusKey key) {
        synchronisationStatusManager.syncFinished(directory.getId(), key, Collections.emptyList());
        try {
            syncEventHelper.publishFailedDirectorySynchronisationEvent(this, directory, UNKNOWN_TIME_TAKEN_VALUE);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
