package com.atlassian.crowd.manager.crypto;

import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.embedded.api.DataReEncryptor;
import com.atlassian.crowd.embedded.api.SwitchableEncryptor;
import com.atlassian.crowd.lock.ClusterLockWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;

@Transactional
public class DefaultEncryptionManager implements EncryptionManager {
    public static final String LOCK_NAME = "CROWD_ENCRYPTION";

    private static final Logger log = LoggerFactory.getLogger(DefaultEncryptionManager.class);

    private final Collection<DataReEncryptor> dataReEncryptors;
    private final ClusterLockWrapper clusterLockWrapper;
    private final SwitchableEncryptor switchableEncryptor;

    public DefaultEncryptionManager(Collection<DataReEncryptor> dataReEncryptors, SwitchableEncryptor switchableEncryptor, ClusterLockService clusterLockService) {
        this.dataReEncryptors = Collections.unmodifiableList(new ArrayList<>(dataReEncryptors));
        this.switchableEncryptor = switchableEncryptor;
        this.clusterLockWrapper = new ClusterLockWrapper(() -> clusterLockService.getLockForName(LOCK_NAME));
    }

    private void reEncryptAllPasswords() {
        log.info("Starting re-encryption of passwords in database");
        dataReEncryptors.forEach(DataReEncryptor::reEncrypt);
        log.info("Re-encryption of passwords in database finished");
    }

    @Override
    public void changeEncryptor(String encryptorKey) {
        clusterLockWrapper.run(() -> switchEncryptor(encryptorKey));
    }

    @Override
    public void changeEncryptionKey() {
        clusterLockWrapper.run(() -> {
            switchableEncryptor.changeEncryptionKey();
            reEncryptAllPasswords();
        });
    }


    @Override
    public void disableEncryption() {
        clusterLockWrapper.run(() -> switchEncryptor(null));
    }

    @Override
    public Collection<String> getAvailableEncryptorNames() {
        return switchableEncryptor.getAvailableEncryptorKeys();
    }

    @Override
    public Optional<String> getDefaultEncryptorName() {
        return switchableEncryptor.getCurrentEncryptorKey();
    }

    private void switchEncryptor(String encryptorKey) {
        switchableEncryptor.switchEncryptor(encryptorKey);
        reEncryptAllPasswords();
    }
}
