package com.atlassian.crowd.manager.avatar;

import java.net.URI;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Locale;

import javax.annotation.Nullable;

import com.atlassian.crowd.manager.property.PropertyManager;
import com.atlassian.crowd.model.user.User;

import com.google.common.base.Charsets;

import org.apache.commons.codec.binary.Hex;

/**
 * A provider of avatars from public web services using the
 * <a href="https://en.gravatar.com/site/implement/">Gravatar API</a>.
 */
public class WebServiceAvatarProvider implements AvatarProvider {
    private final PropertyManager propertyManager;
    private final URI base;

    public WebServiceAvatarProvider(PropertyManager propertyManager, URI base) {
        this.propertyManager = propertyManager;
        this.base = base;
    }

    @Override
    @Nullable
    public URI getUserAvatar(User user, int sizeHint) {
        if (propertyManager.isUseWebAvatars()) {
            String email = user.getEmailAddress();
            if (email != null) {
                return gravatarUrlForEmail(email, sizeHint);
            }
        }

        return null;
    }

    @Override
    public URI getHostedUserAvatarUrl(long applicationId, String username, int sizeHint) {
        return null;
    }

    /**
     * An implementation of https://en.gravatar.com/site/implement/hash/ .
     */
    static String hashOfEmail(String address) {
        try {
            MessageDigest digest = MessageDigest.getInstance("md5");

            return new String(Hex.encodeHex(digest.digest(address.toLowerCase(Locale.ROOT).trim().getBytes(Charsets.UTF_8))));
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("MD5 digest unexpectedly not available", e);
        }
    }

    URI gravatarUrlForEmail(String address, int sizeHint) {
        String hash = hashOfEmail(address);

        return base.resolve(hash + "?s=" + sizeHint);
    }
}
