package com.atlassian.crowd.manager.application.search;

import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.collect.ListMultimap;

import java.util.List;

/**
 * A set of querying across a collection of active directories.
 * <p>
 * A strategy is resolved using a {@link SearchStrategyFactory} and can be optimised for specific configurations of
 * directory.
 *
 * @see SearchStrategyFactory
 */
public interface MembershipSearchStrategy {
    /**
     * Searches for direct group relationships in any of the application's active assigned directories.
     * <p>
     * When searching for the groups an entity is a member of,
     * the {@link com.atlassian.crowd.model.application.Application#isMembershipAggregationEnabled() membership aggregation semantic}
     * will determine whether only the owning directory (when {@code false}) or all directories (when {@code true}) will be searched.
     *
     * @param query membership query.
     * @return List of {@link com.atlassian.crowd.model.user.User} entities,
     * {@link com.atlassian.crowd.model.group.Group} entities,
     * {@link String} usernames or {@link String} group names matching the query criteria.
     */
    <T> List<T> searchDirectGroupRelationships(MembershipQuery<T> query);

    /**
     * Searches for direct and indirect (nested) group relationships in any of the application's active assigned directories.
     * <p>
     * If the directory does not support nested groups, this call will be equivalent to {@link com.atlassian.crowd.manager.directory.DirectoryManager#searchDirectGroupRelationships(long, com.atlassian.crowd.search.query.membership.MembershipQuery)}.
     * <p>
     * <b>WARNING: this method could be very slow if the underlying RemoteDirectory does not employ caching.</b>
     * <p>
     * When searching for the groups a user is a member of only the directory of the user (as determined by findUserByName)
     * is searched.  When searching for memberships of a group or groups a group is a member of all directories are searched and the results amalgamated.
     *
     * @param query membership query.
     * @return List of {@link com.atlassian.crowd.model.user.User} entities,
     * {@link com.atlassian.crowd.model.group.Group} entities,
     * {@link String} usernames or {@link String} group names matching the query criteria.
     */
    <T> List<T> searchNestedGroupRelationships(MembershipQuery<T> query);

    /**
     * Searches for direct group relationships in any of the application's active assigned directories.
     * <p>
     * When searching for the groups an entity is a member of,
     * the {@link com.atlassian.crowd.model.application.Application#isMembershipAggregationEnabled() membership aggregation semantic}
     * will determine whether only the owning directory (when {@code false}) or all directories (when {@code true}) will be searched.
     *
     * @param query membership query.
     * @return {@link ListMultimap} where keys are elements of {@link MembershipQuery#getEntityNamesToMatch()}
     * and values are List of {@link com.atlassian.crowd.model.user.User} entities,
     * {@link com.atlassian.crowd.model.group.Group} entities,
     * {@link String} usernames or {@link String} group names matching the query criteria for the given key.
     */
    <T> ListMultimap<String, T> searchDirectGroupRelationshipsGroupedByName(MembershipQuery<T> query);
}
