package com.atlassian.crowd.manager.application.search;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.manager.application.canonicality.CanonicalityChecker;
import com.atlassian.crowd.manager.application.canonicality.SimpleCanonicalityChecker;
import com.atlassian.crowd.manager.application.filtering.AccessFilter;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.base.Preconditions;
import com.google.common.collect.SetMultimap;

import java.util.List;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

/**
 * A in-memory {@link MembershipSearchStrategy} which only return memberships associated with the canonical users directory.
 * <p>
 * This is considered the worse case {@link MembershipSearchStrategy} to use as searching across multiple directories
 * will be done in-memory, potentially consuming a lot of memory. This is the same (in spirit) as what Crowd 2.8
 * and earlier would do by default.
 *
 * @see com.atlassian.crowd.model.application.Application#isMembershipAggregationEnabled()
 * @since 2.9
 */
public class InMemoryNonAggregatingMembershipSearchStrategy extends AbstractInMemoryMembershipSearchStrategy {
    private final CanonicalityChecker canonicalityChecker;

    public InMemoryNonAggregatingMembershipSearchStrategy(final DirectoryManager directoryManager,
                                                          final List<Directory> directories,
                                                          final CanonicalityChecker canonicalityChecker,
                                                          final AccessFilter accessFilter) {
        super(directoryManager, directories, accessFilter);
        this.canonicalityChecker = canonicalityChecker == null ? new SimpleCanonicalityChecker(directoryManager, directories) : canonicalityChecker;
        Preconditions.checkArgument(canonicalityChecker == null || canonicalityChecker.getDirectories().equals(this.directories));
    }

    @Override
    protected CanonicalityChecker getCanonicalityCheckerIfNeeded(MembershipQuery<?> query) {
        return (directories.size() > 1 && query.isFindChildren()) ? canonicalityChecker : null;
    }

    @Override
    protected <T> BiFunction<Directory, MembershipQuery<T>, MembershipQuery<T>> getQueryTransformer(MembershipQuery<T> original) {
        return (directories.size() == 1 || original.isFindChildren()) ? (directory, query) -> query : filterByCanonical(original);
    }

    private <T> BiFunction<Directory, MembershipQuery<T>, MembershipQuery<T>> filterByCanonical(final MembershipQuery<T> original) {
        final SetMultimap<Long, String> canonical = canonicalityChecker.groupByCanonicalId(
                original.getEntityNamesToMatch(), original.getEntityToMatch());

        return (directory, query) -> filterEntityNamesToMatch(query, canonical.get(directory.getId()));
    }

    private <T> MembershipQuery<T> filterEntityNamesToMatch(final MembershipQuery<T> original, final Set<String> filter) {
        final List<String> filteredNames = original.getEntityNamesToMatch().stream()
                .filter(IdentifierUtils.containsIdentifierPredicate(filter))
                .collect(Collectors.toList());
        return original.withEntityNames(filteredNames);
    }
}
