package com.atlassian.crowd.manager.application.filtering;

import com.atlassian.crowd.directory.DirectoryProperties;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.embedded.impl.IdentifierSet;
import com.atlassian.crowd.manager.application.search.DirectoryManagerSearchWrapper;
import com.atlassian.crowd.manager.application.search.DirectoryQueryWithFilter;
import com.atlassian.crowd.manager.application.search.NamesUtil;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.Combine;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.NullRestriction;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Maps;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

public class BaseAccessFilter implements AccessFilter {
    private static final int QUERY_FOR_ALL_USERS_THRESHOLD = 1000;

    private final DirectoryManagerSearchWrapper directoryManagerSearchWrapper;
    private final Application application;
    private final Map<Long, GroupFilter> groupsWithAccess = new HashMap<>();
    private final Map<Long, IdentifierSet> usersWithAccess = new HashMap<>();
    private final int queryForAllUsersThreshold;

    protected BaseAccessFilter(final DirectoryManager directoryManager, final Application application, boolean queryForAllUsers) {
        Preconditions.checkArgument(application.isFilteringUsersWithAccessEnabled() || application.isFilteringGroupsWithAccessEnabled());
        this.directoryManagerSearchWrapper = new DirectoryManagerSearchWrapper(directoryManager);
        this.application = application;
        this.queryForAllUsersThreshold = queryForAllUsers ? -1 : QUERY_FOR_ALL_USERS_THRESHOLD;
    }

    @Override
    public boolean requiresFiltering(Entity entityType) {
        return runForType(entityType, application::isFilteringUsersWithAccessEnabled, application::isFilteringGroupsWithAccessEnabled);
    }

    @Override
    public boolean hasAccess(long directoryId, Entity entity, String name) {
        if (!requiresFiltering(entity)) {
            return true;
        }
        final ApplicationDirectoryMapping mapping = getMappingOrFail(directoryId);
        final Predicate<String> namesFilter = namesFilter(mapping, entity, ImmutableSet.of(name));
        return namesFilter.test(name);
    }

    @Override
    public <T> Optional<DirectoryQueryWithFilter<T>> getDirectoryQueryWithFilter(Directory directory, MembershipQuery<T> query) {
        final ApplicationDirectoryMapping mapping = getMappingOrFail(directory.getId());
        if (mapping.isAllowAllToAuthenticate()
                || !requiresFiltering(query.getEntityToMatch().getEntityType())
                && !requiresFiltering(query.getEntityToReturn().getEntityType())) {
            return Optional.of(new DirectoryQueryWithFilter<>(directory, query, UnaryOperator.identity()));
        } else if (getGroupFilter(mapping).isEmpty()) {
            return Optional.empty();
        } else {
            final DirectoryQueryWithFilter<T> directoryQueryWithFilter =
                    query.isFindChildren() ? filterChildrenQuery(mapping, query) : filterParentsQuery(mapping, query);
            return directoryQueryWithFilter.getMembershipQuery().getEntityNamesToMatch().isEmpty() ?
                    Optional.empty() : Optional.of(directoryQueryWithFilter);
        }
    }

    @Override
    public <T> Optional<DirectoryQueryWithFilter<T>> getDirectoryQueryWithFilter(Directory directory, EntityQuery<T> query) {
        final ApplicationDirectoryMapping mapping = getMappingOrFail(directory.getId());
        if (mapping.isAllowAllToAuthenticate() || !requiresFiltering(query.getEntityDescriptor().getEntityType())) {
            return Optional.of(new DirectoryQueryWithFilter<>(directory, query, UnaryOperator.identity()));
        } else if (getGroupFilter(mapping).isEmpty()) {
            return Optional.empty();
        } else {
            return Optional.of(filterEntityQuery(mapping, query));
        }
    }

    private <T> DirectoryQueryWithFilter<T> filterEntityQuery(ApplicationDirectoryMapping mapping, EntityQuery<T> query) {
        if (query.getSearchRestriction() != null && !(query.getSearchRestriction() instanceof NullRestriction)
                && !DirectoryProperties.cachesAnyUsers(mapping.getDirectory())) {
            // Remote directories do not support complex queries, so we run simple query and filter results.
            return new DirectoryQueryWithFilter<>(mapping.getDirectory(),
                    query.withAllResults(), list -> filter(mapping, query.getEntityDescriptor().getEntityType(), list));
        }
        if (query.getEntityDescriptor().getEntityType() == Entity.USER) {
            final MembershipQuery<T> membershipQuery = QueryBuilder.queryFor(query.getReturnType(), query.getEntityDescriptor())
                    .with(query.getSearchRestriction())
                    .childrenOf(EntityDescriptor.group())
                    .withNames(getGroupFilter(mapping).getAllWithAccess())
                    .startingAt(query.getStartIndex())
                    .returningAtMost(query.getMaxResults());
            return new DirectoryQueryWithFilter<>(mapping.getDirectory(), membershipQuery, UnaryOperator.identity());
        } else {
            final SearchRestriction restriction = Restriction.on(GroupTermKeys.NAME)
                    .exactlyMatchingAny(getGroupFilter(mapping).getAllWithAccess());
            return new DirectoryQueryWithFilter<>(mapping.getDirectory(),
                    query.withSearchRestriction(Combine.optionalAllOf(query.getSearchRestriction(), restriction)),
                    UnaryOperator.identity());
        }
    }

    private <T> DirectoryQueryWithFilter<T> filterChildrenQuery(ApplicationDirectoryMapping mapping, MembershipQuery<T> original) {
        final MembershipQuery<T> filtered = filterToMatch(original, mapping);
        if (requiresFiltering(mapping, original.getEntityToMatch().getEntityType()) || filtered.equals(original)) {
            // If parents are already filtered, children have access as well, no need to filter results.
            return new DirectoryQueryWithFilter<>(mapping.getDirectory(), filtered, UnaryOperator.identity());
        } else { // returnFiltering must be true
            return new DirectoryQueryWithFilter<>(mapping.getDirectory(), original.withAllResults(),
                    list -> filter(mapping, original.getEntityToReturn().getEntityType(), list));
        }
    }

    private <T> DirectoryQueryWithFilter<T> filterParentsQuery(ApplicationDirectoryMapping mapping, MembershipQuery<T> original) {
        final Entity toReturn = original.getEntityToReturn().getEntityType();
        if (requiresFiltering(mapping, toReturn)) {
            return new DirectoryQueryWithFilter<>(mapping.getDirectory(), original.withAllResults(), list -> filter(mapping, toReturn, list));
        } else if (isSimpleUserParentsQuery(original)) {
            // If we return all parents of the user, there is no need to do costly pre-filter, we can
            // just check if the results have any group with access.
            return new DirectoryQueryWithFilter<>(mapping.getDirectory(), original.withAllResults(), allGroupsOrNone(mapping));
        } else {
            return new DirectoryQueryWithFilter<>(mapping.getDirectory(), filterToMatch(original, mapping), UnaryOperator.identity());
        }
    }

    private boolean requiresFiltering(ApplicationDirectoryMapping mapping, Entity entityType) {
        return requiresFiltering(entityType) && !mapping.isAllowAllToAuthenticate();
    }

    private Predicate<String> namesFilter(ApplicationDirectoryMapping mapping, Entity entityType, Collection<String> names) {
        if (mapping.isAllowAllToAuthenticate()) {
            return name -> true;
        } else if (names.isEmpty() || mapping.getAuthorisedGroupNames().isEmpty()) {
            return name -> false;
        }
        return runForType(entityType,
                () -> getUsersWithAccess(mapping, names)::contains,
                () -> getGroupFilter(mapping)::hasAccess);
    }

    private IdentifierSet getUsersWithAccess(ApplicationDirectoryMapping mapping, Collection<String> names) {
        final Long directoryId = mapping.getDirectory().getId();
        if (names.size() >= queryForAllUsersThreshold || usersWithAccess.containsKey(directoryId)) {
            return usersWithAccess.computeIfAbsent(directoryId, id -> computeUsersWithAccess(mapping));
        }
        final MembershipQuery<String> query = QueryBuilder.queryFor(String.class, EntityDescriptor.group())
                .parentsOf(EntityDescriptor.user())
                .withNames(names)
                .returningAtMost(EntityQuery.ALL_RESULTS);
        final ListMultimap<String, String> userGroups =
                directoryManagerSearchWrapper.searchDirectGroupRelationshipsGroupedByName(directoryId, query);
        return new IdentifierSet(Maps.filterValues(userGroups.asMap(), getGroupFilter(mapping)::anyHasAccess).keySet());
    }

    private IdentifierSet computeUsersWithAccess(ApplicationDirectoryMapping mapping) {
        final MembershipQuery<String> membershipQuery = QueryBuilder.queryFor(String.class, EntityDescriptor.user())
                .childrenOf(EntityDescriptor.group())
                .withNames(getGroupFilter(mapping).getAllWithAccess())
                .returningAtMost(EntityQuery.ALL_RESULTS);
        return new IdentifierSet(directoryManagerSearchWrapper.searchDirectGroupRelationships(mapping.getDirectory().getId(), membershipQuery));
    }

    private <T> UnaryOperator<List<T>> allGroupsOrNone(ApplicationDirectoryMapping mapping) {
        final GroupFilter groupFilter = getGroupFilter(mapping);
        return results -> groupFilter.anyHasAccess(NamesUtil.namesOf(results)) ? results : ImmutableList.of();
    }

    @Nonnull
    private ApplicationDirectoryMapping getMappingOrFail(long directoryId) {
        return Objects.requireNonNull(application.getApplicationDirectoryMapping(directoryId));
    }

    private GroupFilter getGroupFilter(ApplicationDirectoryMapping mapping) {
        return groupsWithAccess.computeIfAbsent(mapping.getDirectory().getId(),
                id -> new GroupFilter(directoryManagerSearchWrapper, mapping));
    }

    private <T> MembershipQuery<T> filterToMatch(MembershipQuery<T> original, ApplicationDirectoryMapping mapping) {
        return original.withEntityNames(filter(
                mapping, original.getEntityToMatch().getEntityType(), original.getEntityNamesToMatch()));
    }

    private <T> List<T> filter(ApplicationDirectoryMapping mapping, Entity entityType, Collection<T> entities) {
        final Predicate<String> namesFilter = namesFilter(mapping, entityType, NamesUtil.namesOf(entities));
        return NamesUtil.filterByName(entities, namesFilter);
    }

    private <T> boolean isSimpleUserParentsQuery(MembershipQuery<T> original) {
        return (original.getSearchRestriction() == null || original.getSearchRestriction() instanceof NullRestriction)
                && original.getEntityNamesToMatch().size() == 1
                && original.getEntityToMatch().getEntityType() == Entity.USER;
    }

    private static <T> T runForType(Entity entityType, Supplier<T> supplierWhenUser, Supplier<T> supplierWhenGroup) {
        switch (entityType) {
            case USER:
                return supplierWhenUser.get();
            case GROUP:
                return supplierWhenGroup.get();
            default:
                throw new IllegalArgumentException("Unsupported entity type " + entityType);
        }
    }
}
