package com.atlassian.crowd.manager.application.filtering;


import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.manager.application.search.DirectoryQueryWithFilter;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;

import java.util.Optional;
import java.util.function.UnaryOperator;

/**
 * Helper class for creating {@link AccessFilter}.
 */
public class AccessFilters {
    public static final AccessFilter UNFILTERED = new AccessFilter() {
        @Override
        public boolean requiresFiltering(Entity entity) {
            return false;
        }

        @Override
        public <T> Optional<DirectoryQueryWithFilter<T>> getDirectoryQueryWithFilter(Directory directory, EntityQuery<T> query) {
            return Optional.of(new DirectoryQueryWithFilter<>(directory, query, UnaryOperator.identity()));
        }

        @Override
        public <T> Optional<DirectoryQueryWithFilter<T>> getDirectoryQueryWithFilter(Directory directory, MembershipQuery<T> query) {
            return Optional.of(new DirectoryQueryWithFilter<>(directory, query, UnaryOperator.identity()));
        }

        @Override
        public boolean hasAccess(long directoryId, Entity entity, String name) {
            return true;
        }
    };

    /**
     * Creates {@link AccessFilter} for the given {@code application}.
     *
     * @param directoryManager directory manager that will be queried
     * @param application      application for which access based filtering should be applied
     * @param queryForAllUsers whether all users with access should be fetched in a single query; setting to true makes
     *                         sense when {@link AccessFilter} will be used for multiple queries
     * @return {@link AccessFilter} for the given {@code application}
     */
    public static AccessFilter create(DirectoryManager directoryManager, Application application, boolean queryForAllUsers) {
        if (application.isFilteringUsersWithAccessEnabled() || application.isFilteringGroupsWithAccessEnabled()) {
            return new BaseAccessFilter(directoryManager, application, queryForAllUsers);
        }
        return UNFILTERED;
    }

    private AccessFilters() {
    }
}
