package com.atlassian.crowd.manager.application.canonicality;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.impl.IdentifierSet;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.manager.application.search.DirectoryManagerSearchWrapper;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.NullRestriction;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.LongFunction;

/**
 * Optimized implementation of {@link CanonicalityChecker}.
 * This implementation is efficient for multiple calls, as it pre-fetches data.
 * Limits number of queries to one per directory and entity type combination.
 * Keeps only shadowed names to decrease memory usage.
 * Does not implement {@link CanonicalityChecker#groupByCanonicalId(Set, EntityDescriptor)} - can't be used for
 * parent queries.
 */
public class OptimizedCanonicalityChecker implements CanonicalityChecker {
    private final DirectoryManagerSearchWrapper directoryManagerSearchWrapper;
    private final List<Directory> directories;
    private final Map<EntityDescriptor, Function<String, Long>> entityToShadowingProviders = new HashMap<>();

    public OptimizedCanonicalityChecker(DirectoryManager directoryManager, List<Directory> directories) {
        this.directoryManagerSearchWrapper = new DirectoryManagerSearchWrapper(directoryManager);
        this.directories = ImmutableList.copyOf(directories);
    }

    public OptimizedCanonicalityChecker(DirectoryManager directoryManager,
                                        List<Directory> directories,
                                        Map<EntityDescriptor, LongFunction<Collection<String>>> providers) {
        this.directoryManagerSearchWrapper = new DirectoryManagerSearchWrapper(directoryManager);
        this.directories = ImmutableList.copyOf(directories);
        providers.forEach((entity, provider) -> entityToShadowingProviders.put(entity, createNameToShadowingDirProvider(entity, provider)));
    }

    @Override
    public void removeNonCanonicalEntities(final Multimap<Long, String> allNames, EntityDescriptor entity) {
        final Function<String, Long> nameToShadowingDirProvider = this.entityToShadowingProviders.computeIfAbsent(
                entity, this::createNameToShadowingDirProvider);
        for (final Directory directory : directories) {
            final Long dirId = directory.getId();
            for (final Iterator<String> it = allNames.get(dirId).iterator(); it.hasNext();) {
                final Long shadowedBy = nameToShadowingDirProvider.apply(it.next());
                if (shadowedBy != null && !shadowedBy.equals(dirId)) {
                    it.remove();
                }
            }
        }
    }

    @Override
    public SetMultimap<Long, String> groupByCanonicalId(Set<String> names, EntityDescriptor entity) {
        throw new UnsupportedOperationException();
    }

    private Function<String, Long> createNameToShadowingDirProvider(final EntityDescriptor entity) {
        final boolean isUserQuery = entity.equals(EntityDescriptor.user());
        final EntityQuery<String> query = QueryBuilder.queryFor(String.class, entity, NullRestriction.INSTANCE, 0, EntityQuery.ALL_RESULTS);
        return this.<DirectoryNotFoundException, OperationFailedException>createNameToShadowingDirProvider(entity,
                isUserQuery ? dirId -> directoryManagerSearchWrapper.searchUsers(dirId, query) : dirId -> directoryManagerSearchWrapper.searchGroups(dirId, query));
    }

    private Function<String, Long> createNameToShadowingDirProvider(final EntityDescriptor entity, final LongFunction<Collection<String>> searcher) {
        Preconditions.checkArgument(entity.equals(EntityDescriptor.user()) || entity.equals(EntityDescriptor.group()));
        if (directories.size() <= 1) {
            return name -> null;
        }
        final Map<String, Long> result = new HashMap<>();
        final Set<String> seen = new HashSet<>();
        final Set<String> shadowed = new HashSet<>();
        for (final Directory directory : directories) {
            final IdentifierSet lowerCasedNames = new IdentifierSet(searcher.apply(directory.getId()));
            for (final String lowerCasedName : lowerCasedNames) {
                if (seen.add(lowerCasedName)) {
                    result.put(lowerCasedName, directory.getId());
                } else {
                    shadowed.add(lowerCasedName);
                }
            }
        }
        result.keySet().retainAll(shadowed);
        final Map<String, Long> immutableResult = ImmutableMap.copyOf(result);
        return name -> immutableResult.get(IdentifierUtils.toLowerCase(name));
    }

    public List<Directory> getDirectories() {
        return directories;
    }
}
