package com.atlassian.crowd.manager.application;

import com.atlassian.crowd.directory.DirectoryProperties;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.spi.UserDao;
import com.atlassian.crowd.manager.recovery.RecoveryModeService;
import com.atlassian.crowd.model.application.Application;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class AuthenticationOrderOptimizer {
    private static final Logger logger = LoggerFactory.getLogger(AuthenticationOrderOptimizer.class);

    private final UserDao userDao;

    private final RecoveryModeService recoveryModeService;

    public AuthenticationOrderOptimizer(final UserDao userDao,
                                        final RecoveryModeService recoveryModeService) {
        this.userDao = userDao;
        this.recoveryModeService = recoveryModeService;
    }

    /**
     * Optimize order of directories for authentication flow. Move cached directories that do not contain user to the end of the queue.
     *
     * @param directories ordered list of directories for normal authentication flow
     * @param username    for which authentication will be performed
     * @return list of directories in optimized order
     */
    List<Directory> optimizeDirectoryOrderForAuthentication(final Application application, final List<Directory> directories, final String username) {
        if (!application.isCachedDirectoriesAuthenticationOrderOptimisationEnabled() || directories.isEmpty()) {
            return directories;
        }

        final Set<Long> directoriesContainingUser = userDao.findDirectoryIdsContainingUserName(username);

        final Predicate<Directory> canDirectoryAuthenticationBePostponed = directory ->
                !recoveryModeService.isRecoveryDirectory(directory)
                        && DirectoryProperties.cachesAllUsers(directory)
                        && !directoriesContainingUser.contains(directory.getId());

        final Map<Boolean, List<Directory>> directoryPostponeMap = directories.stream()
                .collect(Collectors.partitioningBy(canDirectoryAuthenticationBePostponed));

        final List<Directory> directoriesToPostpone = directoryPostponeMap.get(true);
        final List<Directory> directoriesToNormallyProceed = directoryPostponeMap.get(false);

        if (!directoriesToPostpone.isEmpty() && !directoriesToNormallyProceed.isEmpty()) {
            logger.debug("Optimizing authentication order for application {} by moving directories {} to the end of the directory queue.", application.getId(), directoriesToPostpone);
        }

        return ImmutableList.<Directory>builder()
                .addAll(directoriesToNormallyProceed)
                .addAll(directoriesToPostpone)
                .build();
    }
}
