package com.atlassian.crowd.manager.application;

import com.atlassian.crowd.directory.DirectoryProperties;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.OperationType;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.embedded.api.UserCapabilities;
import com.atlassian.crowd.embedded.impl.DirectoryUserCapabilities;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.event.EventStore;
import com.atlassian.crowd.event.EventTokenExpiredException;
import com.atlassian.crowd.event.Events;
import com.atlassian.crowd.event.IncrementalSynchronisationNotAvailableException;
import com.atlassian.crowd.event.user.UserAuthenticatedEvent;
import com.atlassian.crowd.event.user.UserAuthenticationFailedInvalidAuthenticationEvent;
import com.atlassian.crowd.exception.ApplicationPermissionException;
import com.atlassian.crowd.exception.BulkAddFailedException;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.NestedGroupsNotSupportedException;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserAlreadyExistsException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.exception.WebhookNotFoundException;
import com.atlassian.crowd.manager.application.canonicality.CanonicalEntityByNameFinder;
import com.atlassian.crowd.manager.application.canonicality.SimpleCanonicalityChecker;
import com.atlassian.crowd.manager.application.filtering.AccessFilter;
import com.atlassian.crowd.manager.application.filtering.AccessFilters;
import com.atlassian.crowd.manager.application.search.GroupSearchStrategy;
import com.atlassian.crowd.manager.application.search.MembershipSearchStrategy;
import com.atlassian.crowd.manager.application.search.SearchStrategyFactory;
import com.atlassian.crowd.manager.application.search.UserSearchStrategy;
import com.atlassian.crowd.manager.avatar.AvatarProvider;
import com.atlassian.crowd.manager.avatar.AvatarReference;
import com.atlassian.crowd.manager.directory.BulkAddResult;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.manager.directory.DirectoryPermissionException;
import com.atlassian.crowd.manager.permission.PermissionManager;
import com.atlassian.crowd.manager.webhook.InvalidWebhookEndpointException;
import com.atlassian.crowd.manager.webhook.WebhookRegistry;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;
import com.atlassian.crowd.model.application.Applications;
import com.atlassian.crowd.model.event.OperationEvent;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserTemplateWithCredentialAndAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.model.webhook.Webhook;
import com.atlassian.crowd.model.webhook.WebhookTemplate;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.fugue.Pair;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Nullable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.atlassian.crowd.embedded.api.Directories.directoryWithIdPredicate;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang3.BooleanUtils.isFalse;
import static org.apache.commons.lang3.BooleanUtils.toBooleanObject;

@Transactional
public class ApplicationServiceGeneric implements ApplicationService {
    private static final Logger logger = LoggerFactory.getLogger(ApplicationServiceGeneric.class);

    private static final Pattern USER_KEY_PATTERN = Pattern.compile("(\\d+):(.+)");
    private static final Set<OperationType> UPDATE_GROUP_PERMISSION = ImmutableSet.of(OperationType.UPDATE_GROUP);
    private static final Set<OperationType> CREATE_AND_UPDATE_GROUP_PERMISSIONS =
            ImmutableSet.of(OperationType.CREATE_GROUP, OperationType.UPDATE_GROUP);
    private final SearchStrategyFactory searchStrategyFactory;

    private static class DirectoryAndGroup {
        final Directory directory;
        final Group group;

        DirectoryAndGroup(Directory directory, Group group) {
            this.directory = directory;
            this.group = group;
        }
    }

    private final DirectoryManager directoryManager;
    private final PermissionManager permissionManager;
    private final EventPublisher eventPublisher;
    private final EventStore eventStore;
    private final WebhookRegistry webhookRegistry;
    private final AvatarProvider avatarProvider;
    private final AuthenticationOrderOptimizer authenticationOrderOptimizer;

    private final Predicate<Directory> supportsNestedGroups = new Predicate<Directory>() {
        @Override
        public boolean apply(Directory directory) {
            try {
                return directoryManager.supportsNestedGroups(directory.getId());
            } catch (DirectoryInstantiationException e) {
                throw new com.atlassian.crowd.exception.runtime.OperationFailedException(e);
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryAccess(e);
            }
        }
    };

    public ApplicationServiceGeneric(final DirectoryManager directoryManager,
                                     final SearchStrategyFactory searchStrategyFactory,
                                     final PermissionManager permissionManager,
                                     final EventPublisher eventPublisher,
                                     final EventStore eventStore,
                                     final WebhookRegistry webhookRegistry,
                                     final AvatarProvider avatarProvider,
                                     final AuthenticationOrderOptimizer authenticationOrderOptimizer) {
        this.directoryManager = checkNotNull(directoryManager);
        this.searchStrategyFactory = checkNotNull(searchStrategyFactory);
        this.permissionManager = checkNotNull(permissionManager);
        this.eventPublisher = eventPublisher;
        this.eventStore = eventStore;
        this.webhookRegistry = webhookRegistry;
        this.authenticationOrderOptimizer = authenticationOrderOptimizer;
        this.avatarProvider = avatarProvider;
    }

    @Override
    public User authenticateUser(final Application application, final String username, final PasswordCredential passwordCredential) throws OperationFailedException, InactiveAccountException, InvalidAuthenticationException, ExpiredCredentialException, UserNotFoundException {
        if (application.getApplicationDirectoryMappings().isEmpty()) {
            throw new InvalidAuthenticationException("Unable to authenticate user as there are no directories mapped to the application " + application.getName());
        }

        OperationFailedException failedException = null;

        final List<Directory> sortedDirectories = authenticationOrderOptimizer.optimizeDirectoryOrderForAuthentication(application, getActiveDirectories(application), username);

        for (final Directory directory : sortedDirectories) {
            final String directoryDescription = directory.getName() + " (" + directory.getId() + ")";
            try {
                logger.debug("Trying to authenticate user {} in directory {} for application {}", username, directoryDescription, application.getName());
                final User user = directoryManager.authenticateUser(directory.getId(), username, passwordCredential);

                logger.debug("Authenticated user {} in directory {} for application {}", username, directoryDescription, application.getName());
                eventPublisher.publish(new UserAuthenticatedEvent(this, directory, application, user));

                return user;
            } catch (final OperationFailedException e) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Failed to authenticate against '" + directoryDescription + "'", e);
                }
                logger.error("Directory '" + directoryDescription + "' is not functional during authentication of '" + username + "'. Skipped.");

                // we only remember the first one since we can throw only one and the first one should be the most important.
                if (failedException == null) {
                    failedException = e;
                }
            } catch (final UserNotFoundException e) {
                // om nom nom..eat this exception..this can happen if the user does not exist in this directory, so just skip over to the next directory
                logger.debug("User {} does not exist in directory {}, continuing", username, directoryDescription);
            } catch (final InvalidAuthenticationException e) {
                logger.info("Invalid credentials for user {} in directory {}, aborting", username, directoryDescription);
                eventPublisher.publish(new UserAuthenticationFailedInvalidAuthenticationEvent(this, directory, username));
                throw e;
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryIteration(e);
            }
        }

        // at the stage, if the user did not get authenticated by any directory and an OperationFailedException was recorded,
        // we throw that OperationFailedException since it's the prime suspect that should have been the cause of the failed authentication.
        if (failedException != null) {
            logger.debug("Failed to find user {} in any directory for application {}, rethrowing {}", username, application.getName(), failedException);
            throw failedException;
        }

        // otherwise, simply the user does not exist in any of the directories.
        logger.debug("Failed to find user {} in any directory for application {}", username, application.getName());
        throw new UserNotFoundException(username);
    }

    @Override
    public boolean isUserAuthorised(final Application application, final String username) {
        try {
            final User user = findUserByName(application, username);
            return isUserAuthorised(application, user);
        } catch (UserNotFoundException e) {
            return false;
        }
    }

    @Override
    public boolean isUserAuthorised(final Application application, final User user) {
        return isUserAuthorised(user.getName(), user.getDirectoryId(), application);
    }

    private boolean isUserAuthorised(final String username, final long directoryId, final Application application) {
        try {
            // only cache the result of the authorisation if the user exists and there are no operation failures
            return isAllowedToAuthenticate(username, directoryId, application);
        } catch (OperationFailedException e) {
            logger.error(e.getMessage(), e);
            return false;
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }
    }

    @Override
    public void addAllUsers(final Application application, final Collection<UserTemplateWithCredentialAndAttributes> userTemplates) throws ApplicationPermissionException, OperationFailedException, BulkAddFailedException {
        logger.debug("Adding users for application {}", application);

        // if the user doesn't exist in ANY of the directories,
        // add the principal to the FIRST directory with CREATE_USER permission
        final Set<String> failedUsers = new HashSet<>();
        final Set<String> existingUsers = new HashSet<>();

        // iterate through directories, find the first one with CREATE_USER permission
        final Directory directory = findFirstDirectoryWithCreateUserPermission(application);
        if (directory == null) {
            // None have the required permission
            throw new ApplicationPermissionException("Application '" + application.getName() + "' has no directories that allow adding of users.");
        }
        final BulkAddResult<User> result;
        try {
            // Set the Directory ID in each of the UserTemplates
            for (final UserTemplateWithCredentialAndAttributes userTemplate : userTemplates) {
                userTemplate.setDirectoryId(directory.getId());
            }

            result = directoryManager.addAllUsers(directory.getId(), userTemplates, false);
            for (final User user : result.getExistingEntities()) {
                existingUsers.add(user.getName());
            }

            for (final User user : result.getFailedEntities()) {
                failedUsers.add(user.getName());
            }
        } catch (final DirectoryPermissionException ex) {
            // PermissionManager said we had CREATE_USER permission, but the Directory threw a PermissionException
            throw new ApplicationPermissionException(
                    "Permission Exception when trying to add users to directory '" + directory.getName() + "'. " + ex.getMessage(), ex);
        } catch (final DirectoryNotFoundException ex) {
            // The Directory has disappeared from underneath us - smells like concurrent modification.
            throw new OperationFailedException("Directory Not Found when trying to add users to directory '" + directory.getName() + "'.", ex);
        }

        if ((failedUsers.size() > 0) || (existingUsers.size() > 0)) {
            throw new BulkAddFailedException(failedUsers, existingUsers);
        }
    }

    /**
     * Returns the first directory with CREATE_USER permission, or null if none have CREATE_USER permission.
     *
     * @param application the application to explore for DirectoryMappings
     * @return the first directory with CREATE_USER permission, or null if none have CREATE_USER permission.
     */
    private Directory findFirstDirectoryWithCreateUserPermission(final Application application) {
        for (final Directory directory : getActiveDirectories(application)) {
            if (permissionManager.hasPermission(application, directory, OperationType.CREATE_USER)) {
                return directory;
            }
        }
        return null;
    }

    @Override
    public User findUserByName(final Application application, final String name) throws UserNotFoundException {
        return finder(application).findUserByName(name);
    }

    @Override
    public User findRemoteUserByName(Application application, String username) throws UserNotFoundException {
        return finder(application).findRemoteUserByName(username);
    }

    /**
     * Extracts the directory ID and the directory-unique externalId from the key.
     *
     * @param key the user key that is unique in this server
     * @return a pair of the directory ID and the external ID of the user
     * @throws IllegalArgumentException if the key cannot be matched
     */
    @VisibleForTesting
    static Pair<Long, String> directoryIdAndExternalIdFromKey(String key) {
        Matcher matcher = USER_KEY_PATTERN.matcher(key);
        Preconditions.checkArgument(matcher.matches(), "Invalid user key");
        return Pair.pair(Long.parseLong(matcher.group(1)), matcher.group(2));
    }

    @Override
    public User findUserByKey(final Application application,
                              final String key) throws UserNotFoundException {
        Pair<Long, String> directoryIdAndExternalId = directoryIdAndExternalIdFromKey(key);
        long directoryId = directoryIdAndExternalId.left();
        String externalId = directoryIdAndExternalId.right();

        boolean isDirectoryActive = Iterables.any(getActiveDirectories(application), directoryWithIdPredicate(directoryId));
        if (!isDirectoryActive) {
            logger.debug("Cannot look up in directory {} because it is not mapped to the application", directoryId);
            throw new UserNotFoundException(externalId);
        }

        try {
            User user = directoryManager.findUserByExternalId(directoryId, externalId);
            return checkCanonicalUser(user, application);
        } catch (final DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        } catch (final OperationFailedException e) {
            throw new UserNotFoundException(externalId, e);
        }
    }

    @Override
    public UserWithAttributes findUserWithAttributesByKey(final Application application,
                                                          final String key) throws UserNotFoundException {
        Pair<Long, String> directoryIdAndExternalId = directoryIdAndExternalIdFromKey(key);
        long directoryId = directoryIdAndExternalId.left();
        String externalId = directoryIdAndExternalId.right();

        boolean isDirectoryActive = Iterables.any(getActiveDirectories(application), directoryWithIdPredicate(directoryId));
        if (!isDirectoryActive) {
            logger.debug("Cannot look up in directory {} because it is not mapped to the application", directoryId);
            throw new UserNotFoundException(externalId);
        }

        try {
            UserWithAttributes user = directoryManager.findUserWithAttributesByExternalId(directoryId, externalId);
            return checkCanonicalUser(user, application);
        } catch (final DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        } catch (final OperationFailedException e) {
            throw new UserNotFoundException(externalId, e);
        }
    }

    /**
     * Checks if the given user is the canonical user for the given application, and fails if it is not.
     *
     * @param user        a user who may be canonical or shadowed
     * @param application the application to which the user is mapped
     * @param <T>         the type of the user
     * @return the given user, if the user is canonical.
     * @throws UserNotFoundException if the given user is not canonical, i.e., it is shadowed by another user
     * @see #isCanonical
     */
    private <T extends User> T checkCanonicalUser(T user, Application application) throws UserNotFoundException {
        if (!isCanonical(application, user)) {
            logger.debug("Skipping user {} from directory {} because it is shadowed by another user",
                    user.getName(), user.getDirectoryId());
            throw new UserNotFoundException(user.getExternalId());
        } else if (!AccessFilters.create(directoryManager, application, false)
                .hasAccess(user.getDirectoryId(), Entity.USER, user.getName())) {
            throw new UserNotFoundException(user.getExternalId());
        }
        return user;
    }

    /**
     * This method is exactly like findUserByName except that it does not keep cycling if
     * a directory throws an OperationFailedException when attempting to find the user in
     * the particular directory. Use in mutation operations.
     *
     * @param application to get directories for.
     * @param name        of the user to find.
     * @return the user from the first directory that contains it.
     * @throws OperationFailedException if any directory fails to perform the find operation.
     * @throws UserNotFoundException    if none of the directories contain the user.
     */
    private User fastFailingFindUser(final Application application, final String name) throws UserNotFoundException, OperationFailedException {
        return finder(application).fastFailingFindUserByName(name);
    }

    /**
     * This method is exactly like findGroupByName except that it does not keep cycling if
     * a directory throws an OperationFailedException when attempting to find the user in
     * the particular directory. Use in mutation operations.
     *
     * @param application to get directories for.
     * @param name        of the group to find.
     * @return the group from the first directory that contains it.
     * @throws OperationFailedException if any directory fails to perform the find operation.
     * @throws GroupNotFoundException   if none of the directories contain the group.
     */
    private Group fastFailingFindGroup(final Application application, final String name) throws GroupNotFoundException, OperationFailedException {
        return finder(application).fastFailingFindGroupByName(name);
    }

    @Override
    public UserWithAttributes findUserWithAttributesByName(final Application application, final String name) throws UserNotFoundException {
        return finder(application).findUserWithAttributesByName(name);
    }

    @Override
    public User addUser(final Application application, final UserTemplate user, final PasswordCredential credential)
            throws InvalidUserException, OperationFailedException, InvalidCredentialException, ApplicationPermissionException {
        return addUser(application, UserTemplateWithAttributes.toUserWithNoAttributes(user), credential);
    }

    @Override
    public UserWithAttributes addUser(final Application application, final UserTemplateWithAttributes user, final PasswordCredential credential)
            throws InvalidUserException, OperationFailedException, InvalidCredentialException, ApplicationPermissionException {
        if (IdentifierUtils.hasLeadingOrTrailingWhitespace(user.getName())) {
            throw new InvalidUserException(user, "User name may not contain leading or trailing whitespace");
        }

        logger.debug("Adding user <{}> for application <{}>", user.getName(), application.getName());

        try {
            // see if user already exists in any of the directories
            fastFailingFindUser(application, user.getName());
            // CWD-4487: would have made sense to throw UserAlreadyExistsException, but unfortunately this is now API
            throw new InvalidUserException(user, "User already exists");
        } catch (final UserNotFoundException e) {
            // Good - the user doesn't currently exist in ANY of the directories.
        }

        // Add the user to the first directory with ADD permission.
        final Directory directory = findFirstDirectoryWithCreateUserPermission(application);
        if (directory == null) {
            // None have the required permission
            throw new ApplicationPermissionException("Application '" + application.getName() + "' has no directories that allow adding of users.");
        }

        try {
            user.setDirectoryId(directory.getId());
            final UserWithAttributes newUser = directoryManager.addUser(directory.getId(), user, credential);
            logger.debug("User '{}' was added to directory '{}'.", user.getName(), directory.getName(), user.getName());
            return newUser;
        } catch (final DirectoryPermissionException dpe) {
            // permissionManager said we had CREATE_USER permission, but the Directory threw a PermissionException
            throw new ApplicationPermissionException(
                    "Permission Exception when trying to add user '" + user.getName() + "' to directory '" + directory.getName() + "'. " + dpe.getMessage(),
                    dpe);
        } catch (final DirectoryNotFoundException de) {
            // The Directory has disappeared from underneath us - smells like concurrent modification.
            throw new OperationFailedException(
                    "Directory not found when trying to add user '" + user.getName() + "' to directory '" + directory.getName() + "'.", de);
        } catch (UserAlreadyExistsException e) {
            throw new InvalidUserException(user, "User " + user.getName() + " already exists.", e);
        }
    }

    @Override
    public User updateUser(final Application application, final UserTemplate user) throws InvalidUserException, OperationFailedException, ApplicationPermissionException, UserNotFoundException {
        logger.debug("Updating user <{}> for application <{}>", user.getName(), application.getName());

        // Eagerly throw a UserNotFoundException if the user does not exist
        final User existingUser = fastFailingFindUser(application, user.getName());

        //set the external Id if it was not passed to service
        if (StringUtils.isBlank(user.getExternalId())) {
            user.setExternalId(existingUser.getExternalId());
        }

        // Check if this existingUser is in the directory that the caller expected.
        if (user.getDirectoryId() <= 0) {
            // Caller didn't specify a directory - set this one
            user.setDirectoryId(existingUser.getDirectoryId());
        } else {
            // Caller specified a directory - verify it is the same one we just found.
            if (user.getDirectoryId() != existingUser.getDirectoryId()) {
                // Not good - passed user has a different ID to the existing user in this application with given username.
                throw new InvalidUserException(
                        user,
                        "Attempted to update user '" + user.getName() + "' with invalid directory ID " + user.getDirectoryId() + ", we expected ID " + existingUser.getDirectoryId() + ".");
            }
        }

        final Directory directory = findDirectoryById(existingUser.getDirectoryId());

        if (!permissionManager.hasPermission(application, directory, OperationType.UPDATE_USER)) {
            throw new ApplicationPermissionException(
                    "Cannot update user '" + user.getName() + "' because directory '" + directory.getName() + "' does not allow updates.");
        }

        try {
            return directoryManager.updateUser(directory.getId(), user);
        } catch (final DirectoryPermissionException dpe) {
            // permissionManager said we had CREATE_USER permission, but the Directory threw a PermissionException
            throw new ApplicationPermissionException(
                    "Permission Exception when trying to update user '" + user.getName() + "' in directory '" + directory.getName() + "'.", dpe);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }
    }

    @Override
    public User renameUser(final Application application, final String oldUserName, final String newUsername) throws UserNotFoundException, OperationFailedException, ApplicationPermissionException, InvalidUserException {
        logger.debug("Renaming user <{}> to <{}> for application <{}>", oldUserName, newUsername, application.getName());

        // Eagerly throw a UserNotFoundException if the user does not exist
        final User existingUser = fastFailingFindUser(application, oldUserName);

        final Directory directory = findDirectoryById(existingUser.getDirectoryId());

        if (!permissionManager.hasPermission(application, directory, OperationType.UPDATE_USER)) {
            throw new ApplicationPermissionException(
                    "Cannot rename user '" + oldUserName + "' because directory '" + directory.getName() + "' does not allow updates.");
        }

        try {
            return directoryManager.renameUser(directory.getId(), oldUserName, newUsername);
        } catch (final DirectoryPermissionException dpe) {
            // permissionManager said we had CREATE_USER permission, but the Directory threw a PermissionException
            throw new ApplicationPermissionException(
                    "Permission Exception when trying to rename user '" + oldUserName + "' in directory '" + directory.getName() + "'.", dpe);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        } catch (UserAlreadyExistsException e) {
            throw new InvalidUserException(existingUser, "User " + newUsername + " already exists.");
        }
    }

    @Override
    public void updateUserCredential(final Application application, final String username, final PasswordCredential credential) throws OperationFailedException, InvalidCredentialException, ApplicationPermissionException, UserNotFoundException {
        final User user = fastFailingFindUser(application, username);

        final Directory directory = findDirectoryById(user.getDirectoryId());

        if (permissionManager.hasPermission(application, directory, OperationType.UPDATE_USER)) {
            try {
                directoryManager.updateUserCredential(user.getDirectoryId(), username, credential);
            } catch (final DirectoryPermissionException e) {
                // Shouldn't happen because we just checked the permission
                throw new ApplicationPermissionException(e);
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryAccess(e);
            }
        } else {
            // Permission denied
            throw new ApplicationPermissionException(
                    "Not allowed to update user '" + user.getName() + "' in directory '" + directory.getName() + "'.");
        }
    }

    @Override
    public void storeUserAttributes(final Application application, final String username, final Map<String, Set<String>> attributes) throws OperationFailedException, ApplicationPermissionException, UserNotFoundException {
        logger.debug("Storing user attributes for user <{}> and application <{}>", username, application.getName());

        // Find the user (or throw UserNotFoundException)
        final User user = fastFailingFindUser(application, username);

        final Directory directory = findDirectoryById(user.getDirectoryId());

        if (!permissionManager.hasPermission(application, directory, OperationType.UPDATE_USER_ATTRIBUTE)) {
            throw new ApplicationPermissionException(
                    "Not allowed to update user attributes '" + user.getName() + "' in directory '" + directory.getName() + "'.");
        }

        try {
            directoryManager.storeUserAttributes(directory.getId(), username, attributes);
        } catch (final DirectoryPermissionException ex) {
            // Shouldn't happen because we just checked the permission
            throw new ApplicationPermissionException(ex);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }
    }

    @Override
    public void removeUserAttributes(final Application application, final String username, final String attributeName) throws OperationFailedException, ApplicationPermissionException, UserNotFoundException {
        logger.debug("Removing user attributes for user <{}> and application <{}>", username, application.getName());

        // Find the user (or throw UserNotFoundException)
        final User user = fastFailingFindUser(application, username);

        final Directory directory = findDirectoryById(user.getDirectoryId());

        if (!permissionManager.hasPermission(application, directory, OperationType.UPDATE_USER_ATTRIBUTE)) {
            // Permission denied
            throw new ApplicationPermissionException(
                    "Not allowed to update user attributes '" + user.getName() + "' in directory '" + directory.getName() + "'.");
        }

        try {
            directoryManager.removeUserAttributes(directory.getId(), username, attributeName);
        } catch (final DirectoryPermissionException ex) {
            // Shouldn't happen because we just checked the permission
            throw new ApplicationPermissionException(ex);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }
    }

    @Override
    public void removeUser(final Application application, final String username) throws OperationFailedException, ApplicationPermissionException, UserNotFoundException {
        // Find the user (or throw UserNotFoundException)
        final User user = fastFailingFindUser(application, username);

        final Directory directory = findDirectoryById(user.getDirectoryId());

        if (!permissionManager.hasPermission(application, directory, OperationType.DELETE_USER)) {
            // Permission denied
            throw new ApplicationPermissionException(
                    "Not allowed to delete user '" + user.getName() + "' from directory '" + directory.getName() + "'.");
        }

        try {
            directoryManager.removeUser(directory.getId(), username);
        } catch (final DirectoryPermissionException ex) {
            // Shouldn't happen because we just checked the permission
            throw new ApplicationPermissionException(ex);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }
    }

    @Override
    public <T> List<T> searchUsers(final Application application, final EntityQuery<T> query) {
        return getUserSearchStrategyOrFail(application).searchUsers(query);
    }

    ///////////////////// GROUP OPERATIONS /////////////////////

    @Override
    public Group findGroupByName(final Application application, final String name) throws GroupNotFoundException {
        return finder(application).findGroupByName(name);
    }

    @Override
    public GroupWithAttributes findGroupWithAttributesByName(final Application application, final String name) throws GroupNotFoundException {
        return finder(application).findGroupWithAttributesByName(name);
    }

    private CanonicalEntityByNameFinder finder(final Application application) {
        return new CanonicalEntityByNameFinder(directoryManager, getActiveDirectories(application),
                AccessFilters.create(directoryManager, application, false));
    }

    @Override
    public Group addGroup(final Application application, final GroupTemplate group)
            throws InvalidGroupException, OperationFailedException, ApplicationPermissionException {
        if (IdentifierUtils.hasLeadingOrTrailingWhitespace(group.getName())) {
            throw new InvalidGroupException(group, "Group name may not contain leading or trailing whitespace");
        }

        logger.debug("Adding group <{}> for application <{}>", group.getName(), application.getName());

        try {
            // see if group already exists in any of the directories
            fastFailingFindGroup(application, group.getName());
            throw new InvalidGroupException(group, "Group already exists");
        } catch (final GroupNotFoundException e) {
            // if the group doesn't exist in ANY of the directories,
            // add the group to ALL of the directories with ADD permission
            final OperationType operationType = getCreateOperationType(group);

            // iterate through directories, try to add to all
            for (final Directory directory : getActiveDirectories(application)) {
                if (permissionManager.hasPermission(application, directory, operationType)) {
                    try {
                        group.setDirectoryId(directory.getId());
                        directoryManager.addGroup(directory.getId(), group);
                    } catch (final DirectoryPermissionException dpe) {
                        // this is a legitimate alternate-flow, skip over this directory
                        logger.info("Could not add group <{}> to directory <{}>", group.getName(), directory.getName());
                        logger.info(dpe.getMessage());
                    } catch (final DirectoryNotFoundException onfe) {
                        // Log the error and keep trying, the given directory could not be found
                        logger.error(onfe.getMessage(), onfe);
                    }
                }
            }
        }

        try {
            // return the added group by finding it
            return fastFailingFindGroup(application, group.getName());
        } catch (final GroupNotFoundException e) {
            // no application/directory had add permissions
            throw new ApplicationPermissionException("Application \"" + application.getName() + "\" does not allow adding of groups");
        }
    }

    @Override
    public Group updateGroup(final Application application, final GroupTemplate group)
            throws InvalidGroupException, OperationFailedException, ApplicationPermissionException, GroupNotFoundException {
        logger.debug("Updating group <{}> for application <{}>", group.getName(), application.getName());

        // make sure group exists in at least one of the directories
        final Group groupToUpdate = fastFailingFindGroup(application, group.getName());
        final OperationType operationType = getUpdateOperationType(groupToUpdate);

        boolean atleastOneDirectoryHasPermission = false;

        // iterate through directories, try to add to all
        for (final Directory directory : getActiveDirectories(application)) {
            if (permissionManager.hasPermission(application, directory, operationType)) {
                try {
                    group.setDirectoryId(directory.getId());
                    directoryManager.updateGroup(directory.getId(), group);

                    atleastOneDirectoryHasPermission = true;
                } catch (final DirectoryPermissionException dpe) {
                    // this is a legitimate alternate-flow, skip over this directory
                    logger.info("Could not update group <{}> to directory <{}>", group.getName(), directory.getName());
                    logger.info(dpe.getMessage());
                } catch (final GroupNotFoundException e) {
                    // group does not exist in this directory, skip
                } catch (DirectoryNotFoundException e) {
                    throw concurrentModificationExceptionForDirectoryIteration(e);
                } catch (ReadOnlyGroupException e) {
                    logger.info("Could not update group <{}> to directory <{}> because the group is read-only.",
                            group.getName(), directory.getName(), e);
                }
            }
        }

        if (!atleastOneDirectoryHasPermission) {
            throw new ApplicationPermissionException("Application \"" + application.getName() + "\" does not allow group modifications");
        }

        // return the updated group by finding it
        return fastFailingFindGroup(application, group.getName());
    }

    @Override
    public void storeGroupAttributes(final Application application, final String groupname, final Map<String, Set<String>> attributes)
            throws OperationFailedException, ApplicationPermissionException, GroupNotFoundException {
        logger.debug("Storing group attributes for group <{}> and application <{}>", groupname, application.getName());

        // make sure group exists in at least one of the directories
        final Group groupToUpdate = fastFailingFindGroup(application, groupname);
        final OperationType operationType = getUpdateAttributeOperationType(groupToUpdate);

        boolean atleastOneDirectoryHasPermission = false;

        // iterate through directories, try to add to all
        for (final Directory directory : getActiveDirectories(application)) {
            if (permissionManager.hasPermission(application, directory, operationType)) {
                try {
                    directoryManager.storeGroupAttributes(directory.getId(), groupname, attributes);

                    atleastOneDirectoryHasPermission = true;
                } catch (final DirectoryPermissionException dpe) {
                    // this is a legitimate alternate-flow, skip over this directory
                    logger.info("Could not update group <{}> to directory <{}>", groupname, directory.getName());
                    logger.info(dpe.getMessage());
                } catch (final GroupNotFoundException e) {
                    // group does not exist in this directory, skip
                } catch (DirectoryNotFoundException e) {
                    throw concurrentModificationExceptionForDirectoryIteration(e);
                }
            }
        }

        if (!atleastOneDirectoryHasPermission) {
            throw new ApplicationPermissionException("Application \"" + application.getName() + "\" does not allow group attribute modifications");
        }
    }

    @Override
    public void removeGroupAttributes(final Application application, final String groupname, final String attributeName)
            throws OperationFailedException, ApplicationPermissionException, GroupNotFoundException {
        logger.debug("Removing group attributes for group <{}> and application <{}>", groupname, application.getName());

        // make sure group exists in at least one of the directories
        final Group groupToUpdate = fastFailingFindGroup(application, groupname);

        boolean atleastOneDirectoryHasPermission = false;

        final OperationType operationType = getUpdateAttributeOperationType(groupToUpdate);

        // iterate through directories, try to add to all
        for (final Directory directory : getActiveDirectories(application)) {
            if (permissionManager.hasPermission(application, directory, operationType)) {
                try {
                    directoryManager.removeGroupAttributes(directory.getId(), groupname, attributeName);

                    atleastOneDirectoryHasPermission = true;
                } catch (final DirectoryPermissionException dpe) {
                    // this is a legitimate alternate-flow, skip over this directory
                    logger.info("Could not update group <{}> to directory <{}>", groupname, directory.getName());
                    logger.info(dpe.getMessage());
                } catch (final GroupNotFoundException e) {
                    // group does not exist in this directory, skip
                } catch (DirectoryNotFoundException e) {
                    throw concurrentModificationExceptionForDirectoryIteration(e);
                }
            }
        }

        if (!atleastOneDirectoryHasPermission) {
            throw new ApplicationPermissionException("Application \"" + application.getName() + "\" does not allow group attribute modifications");
        }
    }

    @Override
    public void removeGroup(final Application application, final String groupname)
            throws OperationFailedException, ApplicationPermissionException, GroupNotFoundException {
        // eagerly throw GroupNotFoundException
        final Group groupToRemove = fastFailingFindGroup(application, groupname);

        boolean permissibleByAnyDirectory = false;

        final OperationType operationType = getDeleteOperationType(groupToRemove);

        // remove each group
        for (final Directory directory : getActiveDirectories(application)) {
            if (permissionManager.hasPermission(application, directory, operationType)) {
                try {
                    directoryManager.removeGroup(directory.getId(), groupname);
                    permissibleByAnyDirectory = true;
                } catch (final DirectoryPermissionException e) {
                    // this is a legitimate alternate-flow, skip over this directory
                    logger.info("Could not remove group <{}> from directory <{}>", groupname, directory.getName());
                } catch (final GroupNotFoundException e) {
                    // this directory does not have the group, skip
                } catch (DirectoryNotFoundException e) {
                    throw concurrentModificationExceptionForDirectoryIteration(e);
                } catch (ReadOnlyGroupException e) {
                    logger.info("Could not update group <{}> to directory <{}> because the group is read-only.",
                            groupname, directory.getName(), e);
                }
            }
        }

        if (!permissibleByAnyDirectory) {
            // no directory had remove permissions
            throw new ApplicationPermissionException("Application \"" + application.getName() + "\" does not allow group removal");
        }

    }

    @Override
    public <T> List<T> searchGroups(final Application application, final EntityQuery<T> query) {
        return getGroupSearchStrategyOrFail(application).searchGroups(query);
    }

    ///////////////////// MEMBERSHIP OPERATIONS /////////////////////

    @Override
    public void addUserToGroup(final Application application, final String username, final String groupName)
            throws OperationFailedException, ApplicationPermissionException, UserNotFoundException, GroupNotFoundException,
            MembershipAlreadyExistsException {
        final Directory directory = application.isMembershipAggregationEnabled() ?
                findDirectoryToAddUserToGroupAggregating(application, username, groupName) :
                findDirectoryToAddUserToGroupNonAggregating(application, username, groupName);

        try {
            directoryManager.addUserToGroup(directory.getId(), username, groupName);
        } catch (final DirectoryPermissionException e) {
            // Should not occur as we already checked. Probable causes include an unfortunately-timed permissions
            // change, or a misconfiguration of Crowd.
            throw new ApplicationPermissionException(
                    "Permission Exception when trying to update group '" + groupName + "' in directory '" + directory.getName() + "'.", e);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        } catch (ReadOnlyGroupException e) {
            throw new ApplicationPermissionException(String.format(
                    "Could not add user %s to group %s in directory %s because the directory or group is read-only.",
                    username, groupName, directory.getName()));
        }
    }

    /**
     * Find the directory that contains the canonical user, if it contains the group. If not, and the group exists,
     * try to create the group in the canonical user's directory and return that directory.
     */
    private Directory findDirectoryToAddUserToGroupNonAggregating(Application application, String username, String groupName)
            throws UserNotFoundException, OperationFailedException, GroupNotFoundException, ApplicationPermissionException {
        final long userDirectoryId = fastFailingFindUser(application, username).getDirectoryId();
        // try to find this group within the User's directory
        try {
            try {
                directoryManager.findGroupByName(userDirectoryId, groupName);
            } catch (GroupNotFoundException e) {
                // Try to add the group to the directory with the user, else fail
                Group group = fastFailingFindGroup(application, groupName);

                try {
                    directoryManager.addGroup(userDirectoryId, new GroupTemplate(group).withDirectoryId(userDirectoryId));
                } catch (InvalidGroupException e1) {
                    throw new GroupNotFoundException(
                            String.format(
                                    "Unable to create group %s in directory %d in order to add membership of user %s" +
                                            " (group %s found in directory %d)",
                                    group.getName(), userDirectoryId, username, group.getName(), userDirectoryId
                            ),
                            e1);
                } catch (DirectoryPermissionException e1) {
                    throw new ApplicationPermissionException(e1);
                }

            }
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }

        final Directory directory = findDirectoryById(userDirectoryId);
        if (!hasPermissions(application, UPDATE_GROUP_PERMISSION).apply(directory)) {
            throw new ApplicationPermissionException(
                    "Cannot update group '" + groupName + "' because directory '" + directory.getName() + "' does not allow updates.");
        }

        return directory;
    }

    /**
     * Find the first directory that is writeable and contains both the user and the group. If the group exists and the
     * user exists in a writeable directory, try to create the group in that directory and return that.
     */
    private Directory findDirectoryToAddUserToGroupAggregating(Application application, String username, String groupName)
            throws UserNotFoundException, MembershipAlreadyExistsException, GroupNotFoundException,
            OperationFailedException, ApplicationPermissionException {
        final Collection<Directory> directoriesWithUser =
                ImmutableList.copyOf(Iterables.filter(getActiveDirectories(application), containsUser(username)));

        if (directoriesWithUser.isEmpty()) {
            throw new UserNotFoundException(username);
        }

        final Collection<Directory> directoriesWithUserAndGroup =
                ImmutableList.copyOf(Iterables.filter(directoriesWithUser, containsGroup(groupName)));

        if (Iterables.any(directoriesWithUserAndGroup, containsUserDirectMembershipInGroup(username, groupName))) {
            throw new MembershipAlreadyExistsException(username, groupName);
        } else {
            final Directory writeableDirectoryWithUserAndGroup = directoryWithPermissions(application,
                    directoriesWithUserAndGroup,
                    UPDATE_GROUP_PERMISSION);

            final Directory directory;

            if (writeableDirectoryWithUserAndGroup != null) {
                directory = writeableDirectoryWithUserAndGroup;
            } else {
                // Create the group and then form the membership, so we need CREATE_GROUP permission
                directory = directoryWithPermissions(application,
                        directoriesWithUser,
                        CREATE_AND_UPDATE_GROUP_PERMISSIONS);
                if (directory != null) {
                    final Group group = fastFailingFindGroup(application, groupName);

                    try {
                        directoryManager.addGroup(directory.getId(), new GroupTemplate(group).withDirectoryId(directory.getId()));
                    } catch (InvalidGroupException e) {
                        throw new OperationFailedException(e);
                    } catch (DirectoryNotFoundException e) {
                        throw concurrentModificationExceptionForDirectoryAccess(e);
                    } catch (DirectoryPermissionException e) {
                        throw new ApplicationPermissionException(e);
                    }
                }
            }

            if (directory == null) {
                throw new ApplicationPermissionException(
                        "Did not have update groups permission in any of the directories " + directoriesWithUserAndGroup);
            }
            return directory;
        }
    }

    @Nullable
    private Directory directoryWithPermissionsAnd(Application application,
                                                  Collection<Directory> directories,
                                                  Set<OperationType> permissions,
                                                  Predicate<Directory> andPredicate) {
        return Iterables.find(directories, Predicates.and(hasPermissions(application, permissions), andPredicate), null);
    }

    @Nullable
    private Directory directoryWithPermissions(Application application,
                                               Collection<Directory> directories,
                                               Set<OperationType> permissions) {
        return directoryWithPermissionsAnd(application, directories, permissions, Predicates.<Directory>alwaysTrue());
    }

    private Predicate<Directory> hasPermissions(final Application application, final Set<OperationType> operationTypes) {
        return directory -> Iterables.all(operationTypes,
                operationType -> permissionManager.hasPermission(application, directory, operationType));
    }

    private abstract class DirectoryPredicate implements Predicate<Directory> {
        @Override
        public final boolean apply(Directory directory) {
            try {
                return fallibleCheckForEntity(directory);
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryAccess(e);
            } catch (ObjectNotFoundException e) {
                return false;
            } catch (OperationFailedException e) {
                throw new com.atlassian.crowd.exception.runtime.OperationFailedException(errorMessage(directory), e);
            }
        }

        protected abstract boolean fallibleCheckForEntity(Directory directory)
                throws ObjectNotFoundException, OperationFailedException, DirectoryNotFoundException;

        protected abstract String errorMessage(Directory directory);
    }

    private Predicate<Directory> containsUserDirectMembershipInGroup(final String username, final String groupName) {
        return new DirectoryPredicate() {
            @Override
            protected boolean fallibleCheckForEntity(Directory directory) throws DirectoryNotFoundException, OperationFailedException {
                return directoryManager.isUserDirectGroupMember(directory.getId(), username, groupName);
            }

            @Override
            protected String errorMessage(Directory directory) {
                return String.format("Failed to determine if user %s is a member of group %s in directory %d",
                        username, groupName, directory.getId());
            }
        };
    }

    private Predicate<Directory> containsUserNestedMembershipInGroup(final String username, final String groupName) {
        return new DirectoryPredicate() {
            @Override
            protected boolean fallibleCheckForEntity(Directory directory) throws DirectoryNotFoundException, OperationFailedException {
                return directoryManager.isUserNestedGroupMember(directory.getId(), username, groupName);
            }

            @Override
            protected String errorMessage(Directory directory) {
                return String.format("Failed to determine if user %s is a nested member of group %s in directory %d",
                        username, groupName, directory.getId());
            }
        };
    }

    private Predicate<Directory> containsGroupDirectMembershipInGroup(final String childName, final String parentName) {
        return new DirectoryPredicate() {
            @Override
            protected boolean fallibleCheckForEntity(Directory directory) throws DirectoryNotFoundException, OperationFailedException {
                return directoryManager.isGroupDirectGroupMember(directory.getId(), childName, parentName);
            }

            @Override
            protected String errorMessage(Directory directory) {
                return String.format("Failed to determine if child group %s is a member of parent group %s in directory %d",
                        childName, parentName, directory.getId());
            }
        };
    }

    private Predicate<Directory> containsGroupNestedMembershipInGroup(final String childName, final String parentName) {
        return new DirectoryPredicate() {
            @Override
            protected boolean fallibleCheckForEntity(Directory directory) throws DirectoryNotFoundException, OperationFailedException {
                return directoryManager.isGroupNestedGroupMember(directory.getId(), childName, parentName);
            }

            @Override
            protected String errorMessage(Directory directory) {
                return String.format("Failed to determine if child group %s is a nested member of parent group %s in directory %d",
                        childName, parentName, directory.getId());
            }
        };
    }

    private Predicate<Directory> containsGroup(final String groupName) {
        return new DirectoryPredicate() {
            @Override
            protected boolean fallibleCheckForEntity(Directory directory)
                    throws DirectoryNotFoundException, OperationFailedException, GroupNotFoundException {
                directoryManager.findGroupByName(directory.getId(), groupName);
                return true; // if the group does not exist, it will throw an exception
            }

            @Override
            protected String errorMessage(Directory directory) {
                return String.format("Failed to determine if group %s exists in directory %d",
                        groupName, directory.getId());
            }
        };
    }

    private Predicate<Directory> containsUser(final String username) {
        return new DirectoryPredicate() {
            @Override
            protected boolean fallibleCheckForEntity(Directory directory)
                    throws DirectoryNotFoundException, OperationFailedException, UserNotFoundException {
                directoryManager.findUserByName(directory.getId(), username);
                return true; // if the user does not exist, it will throw an exception
            }

            @Override
            protected String errorMessage(Directory directory) {
                return String.format("Failed to determine if user %s exists in directory %d",
                        username, directory.getId());
            }
        };
    }

    /**
     * Finds the directory. If the directory could not be found, catches DirectoryNotFoundException and throws ConcurrentModificationException
     *
     * @param directoryId The directory id
     * @return the requested directory
     * @throws ConcurrentModificationException when the directory mapping has been concurrently modified
     */
    private Directory findDirectoryById(final long directoryId) throws ConcurrentModificationException {
        try {
            return directoryManager.findDirectoryById(directoryId);
        } catch (final DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }
    }

    @Override
    public void addGroupToGroup(final Application application, final String childGroupName, final String parentGroupName)
            throws OperationFailedException, ApplicationPermissionException, GroupNotFoundException,
            InvalidMembershipException, MembershipAlreadyExistsException {
        final Directory directory;
        @Nullable final Group createdParentGroup;

        // Trivial circular reference: same group
        if (IdentifierUtils.equalsInLowerCase(childGroupName, parentGroupName)) {
            throw new InvalidMembershipException("Cannot add a group to itself.");
        }
        // Check for circular reference - is the "parent" already a member of the "child" in our application?
        // (Note that this could still cause problems in other applications.)
        if (isGroupNestedGroupMember(application, parentGroupName, childGroupName)) {
            throw new InvalidMembershipException(
                    "Cannot add child group '" + childGroupName + "' to parent group '" + parentGroupName + "' - this would cause a circular dependency.");
        }

        final DirectoryAndGroup pair = application.isMembershipAggregationEnabled() ?
                findDirectoryAndGroupForAddGroupToGroupAggregating(application, childGroupName, parentGroupName) :
                findDirectoryAndGroupForAddGroupToGroupNonAggregating(application, childGroupName, parentGroupName);
        directory = pair.directory;
        createdParentGroup = pair.group;

        try {
            final Group childGroup = directoryManager.findGroupByName(directory.getId(), childGroupName);
            final Group parentGroup = createdParentGroup != null ?
                    createdParentGroup : directoryManager.findGroupByName(directory.getId(), parentGroupName);

            if (childGroup.getType() != parentGroup.getType()) {
                throw new InvalidMembershipException(
                        "Cannot add group of type " + childGroup.getType().name() + " to group of type " + parentGroup.getType().name());
            }

            directoryManager.addGroupToGroup(directory.getId(), childGroupName, parentGroupName);
        } catch (final DirectoryPermissionException e) {
            // Should not occur as we already checked. Probable causes include an unfortunately-timed permissions
            // change, or a misconfiguration of Crowd.
            throw new ApplicationPermissionException(
                    "Permission Exception when trying to update group '" + parentGroupName + "' in directory '" + directory.getName() + "'.", e);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        } catch (ReadOnlyGroupException e) {
            throw new ApplicationPermissionException(String.format(
                    "Could not add child group %s to parent group %s in directory %s because the directory or group is read-only.",
                    childGroupName, parentGroupName, directory.getName()));
        } catch (NestedGroupsNotSupportedException e) {
            throw new InvalidMembershipException(e);
        }
    }

    /**
     * Find the directory that contains the canonical child, if it contains the parent or if the parent can be
     * added to that directory.
     */
    private DirectoryAndGroup findDirectoryAndGroupForAddGroupToGroupNonAggregating(Application application,
                                                                                    String childGroupName,
                                                                                    String parentGroupName)
            throws GroupNotFoundException, OperationFailedException, InvalidMembershipException, ApplicationPermissionException {
        @Nullable Group createdParentGroup = null;
        final long childDirectoryId = fastFailingFindGroup(application, childGroupName).getDirectoryId();
        final Directory directory = findDirectoryById(childDirectoryId);

        try {
            if (!directoryManager.supportsNestedGroups(childDirectoryId)) {
                throw new InvalidMembershipException(
                        "Nested directories are not supported by directory " + directory.getName());
            }
            try {
                directoryManager.findGroupByName(childDirectoryId, parentGroupName);
            } catch (GroupNotFoundException e) {
                final Group parentGroup = fastFailingFindGroup(application, parentGroupName);
                createdParentGroup = directoryManager.addGroup(directory.getId(),
                        new GroupTemplate(parentGroup).withDirectoryId(directory.getId()));
            }
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        } catch (InvalidGroupException e) {
            throw new OperationFailedException(e);
        } catch (DirectoryPermissionException e) {
            throw new ApplicationPermissionException(String.format(
                    "Parent group %s could not be added to directory %d where the canonical instance of %s was found.",
                    parentGroupName, childDirectoryId, childGroupName));
        }

        if (!hasPermissions(application, UPDATE_GROUP_PERMISSION).apply(directory)) {
            throw new ApplicationPermissionException(
                    "Cannot update group '" + parentGroupName + "' because directory '" + directory.getName() + "' does not allow updates.");
        }

        return new DirectoryAndGroup(directory, createdParentGroup);
    }

    /**
     * Find the first directory that is writeable and contains both the child and the parent.
     * If this does not exist, fail.
     */
    private DirectoryAndGroup findDirectoryAndGroupForAddGroupToGroupAggregating(Application application,
                                                                                 String childGroupName,
                                                                                 String parentGroupName)
            throws GroupNotFoundException, MembershipAlreadyExistsException, OperationFailedException, ApplicationPermissionException {
        @Nullable final Group createdParentGroup;

        final Collection<Directory> activeDirectories = getActiveDirectories(application);

        final Collection<Directory> directoriesWithChild =
                ImmutableList.copyOf(Iterables.filter(activeDirectories, containsGroup(childGroupName)));
        if (directoriesWithChild.isEmpty()) {
            throw new GroupNotFoundException(childGroupName);
        }

        final Collection<Directory> directoriesWithChildAndParent =
                ImmutableList.copyOf(Iterables.filter(directoriesWithChild, containsGroup(parentGroupName)));

        if (Iterables.any(directoriesWithChildAndParent, containsGroupDirectMembershipInGroup(childGroupName, parentGroupName))) {
            throw new MembershipAlreadyExistsException(childGroupName, parentGroupName);
        } else {
            final Directory writeableDirectoryWithChildAndParent = directoryWithPermissionsAnd(application,
                    directoriesWithChildAndParent,
                    UPDATE_GROUP_PERMISSION,
                    supportsNestedGroups);
            final Directory directory;

            if (writeableDirectoryWithChildAndParent != null) {
                directory = writeableDirectoryWithChildAndParent;
                createdParentGroup = null;
            } else {
                directory = directoryWithPermissionsAnd(application,
                        directoriesWithChild,
                        CREATE_AND_UPDATE_GROUP_PERMISSIONS,
                        supportsNestedGroups);
                if (directory != null) {
                    final Group parentGroup = fastFailingFindGroup(application, parentGroupName);

                    try {
                        createdParentGroup = directoryManager.addGroup(directory.getId(),
                                new GroupTemplate(parentGroup).withDirectoryId(directory.getId()));
                    } catch (InvalidGroupException e) {
                        throw new OperationFailedException(e);
                    } catch (DirectoryNotFoundException e) {
                        throw concurrentModificationExceptionForDirectoryAccess(e);
                    } catch (DirectoryPermissionException e) {
                        throw new ApplicationPermissionException(e);
                    }
                } else {
                    createdParentGroup = null;
                }
            }

            if (directory == null) {
                throw new ApplicationPermissionException(String.format(
                        "Could not find a directory in which it is possible to add %s to %s",
                        childGroupName, parentGroupName));
            }

            return new DirectoryAndGroup(directory, createdParentGroup);
        }
    }

    @Override
    public void removeUserFromGroup(final Application application, final String username, final String groupName)
            throws OperationFailedException, ApplicationPermissionException, MembershipNotFoundException,
            UserNotFoundException, GroupNotFoundException {
        if (application.isMembershipAggregationEnabled()) {
            removeUserFromGroupAggregating(application, username, groupName);
        } else {
            removeUserFromGroupNonAggregating(application, username, groupName);
        }
    }

    /**
     * Remove the membership from the directory containing the canonical user, if it also contains the group.
     */
    private void removeUserFromGroupNonAggregating(Application application, String username, String groupName)
            throws UserNotFoundException, OperationFailedException, GroupNotFoundException, MembershipNotFoundException,
            ApplicationPermissionException {
        final User user = fastFailingFindUser(application, username);

        try {
            directoryManager.findGroupByName(user.getDirectoryId(), groupName);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }

        if (!isUserDirectGroupMember(application, username, groupName)) {
            throw new MembershipNotFoundException(username, groupName);
        }

        Directory directory = findDirectoryById(user.getDirectoryId());
        if (hasPermissions(application, UPDATE_GROUP_PERMISSION).apply(directory)) {
            try {
                directoryManager.removeUserFromGroup(directory.getId(), username, groupName);
            } catch (final DirectoryPermissionException e) {
                throw new ApplicationPermissionException(e);
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryAccess(e);
            } catch (ReadOnlyGroupException e) {
                throw new ApplicationPermissionException(String.format(
                        "Could not remove user %s from group %s in directory %s because the directory or group is read-only.",
                        username, groupName, directory.getName()));
            }
        } else {
            // directory had no remove permissions
            throw new ApplicationPermissionException("Application \"" + application.getName() + "\" does not allow group modifications");
        }
    }

    /**
     * Remove the membership from all directories containing it, if they are all writeable.
     */
    private void removeUserFromGroupAggregating(Application application, String username, String groupName)
            throws UserNotFoundException, GroupNotFoundException, MembershipNotFoundException, OperationFailedException,
            ApplicationPermissionException {
        final Collection<Directory> activeDirectories = getActiveDirectories(application);

        final Collection<Directory> directoriesWithMembership =
                ImmutableList.copyOf(Iterables.filter(activeDirectories, containsUserDirectMembershipInGroup(username, groupName)));

        if (directoriesWithMembership.isEmpty()) {
            if (!Iterables.any(activeDirectories, containsUser(username))) {
                throw new UserNotFoundException(username);
            } else if (!Iterables.any(activeDirectories, containsGroup(groupName))) {
                throw new GroupNotFoundException(groupName);
            } else {
                throw new MembershipNotFoundException(username, groupName);
            }
        } else if (Iterables.all(directoriesWithMembership, hasPermissions(application, UPDATE_GROUP_PERMISSION))) {
            for (Directory directory : directoriesWithMembership) {
                try {
                    directoryManager.removeUserFromGroup(directory.getId(), username, groupName);
                } catch (DirectoryPermissionException e) {
                    throw new ApplicationPermissionException(e);
                } catch (DirectoryNotFoundException e) {
                    // directory is no longer mapped, no need to remove the membership
                } catch (ReadOnlyGroupException e) {
                    throw new ApplicationPermissionException(e);
                }
            }
        } else {
            throw new ApplicationPermissionException(String.format(
                    "At least one directory containing %s as a member of %s does not have write permission",
                    username, groupName));
        }
    }

    @Override
    public void removeGroupFromGroup(final Application application, final String childGroupName, final String parentGroupName)
            throws OperationFailedException, ApplicationPermissionException, MembershipNotFoundException, GroupNotFoundException {
        if (application.isMembershipAggregationEnabled()) {
            removeGroupFromGroupAggregating(application, childGroupName, parentGroupName);
        } else {
            removeGroupFromGroupNonAggregating(application, childGroupName, parentGroupName);
        }
    }

    /**
     * Remove the membership from the directory containing the canonical child group, if it also contains the parent
     * group.
     */
    private void removeGroupFromGroupNonAggregating(Application application, String childGroupName, String parentGroupName)
            throws GroupNotFoundException, OperationFailedException, MembershipNotFoundException, ApplicationPermissionException {
        final Group childGroup = fastFailingFindGroup(application, childGroupName);

        try {
            // Fail if parent group does not exist
            directoryManager.findGroupByName(childGroup.getDirectoryId(), parentGroupName);
        } catch (DirectoryNotFoundException e) {
            throw concurrentModificationExceptionForDirectoryAccess(e);
        }

        if (!isGroupDirectGroupMember(application, childGroupName, parentGroupName)) {
            throw new MembershipNotFoundException(childGroupName, parentGroupName);
        }

        Directory directory = findDirectoryById(childGroup.getDirectoryId());
        if (hasPermissions(application, UPDATE_GROUP_PERMISSION).apply(directory)) {
            try {
                directoryManager.removeGroupFromGroup(directory.getId(), childGroupName, parentGroupName);
            } catch (DirectoryPermissionException e) {
                throw new ApplicationPermissionException(e);
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryAccess(e);
            } catch (ReadOnlyGroupException e) {
                throw new ApplicationPermissionException(
                        String.format(
                                "Could not remove child group %s from parent group %s in directory %s because the directory or group is read-only.",
                                childGroupName, parentGroupName, directory.getName()),
                        e);
            } catch (InvalidMembershipException e) {
                throw new OperationFailedException(
                        String.format(
                                "Cannot remove group %s from %s because they have different types",
                                childGroupName, parentGroupName),
                        e);
            }
        } else {
            // directory had no remove permissions
            throw new ApplicationPermissionException("Application \"" + application.getName() + "\" does not allow group modifications");
        }
    }

    /**
     * Remove the membership from all directories containing it, if they are all writeable.
     */
    private void removeGroupFromGroupAggregating(Application application, String childGroupName, String parentGroupName)
            throws GroupNotFoundException, MembershipNotFoundException, OperationFailedException, ApplicationPermissionException {
        final Collection<Directory> activeDirectories = getActiveDirectories(application);

        final Collection<Directory> directoriesWithMembership =
                ImmutableList.copyOf(Iterables.filter(activeDirectories, containsGroupDirectMembershipInGroup(childGroupName, parentGroupName)));

        if (directoriesWithMembership.isEmpty()) {
            if (!Iterables.any(activeDirectories, containsGroup(childGroupName))) {
                throw new GroupNotFoundException(childGroupName);
            } else if (!Iterables.any(activeDirectories, containsGroup(parentGroupName))) {
                throw new GroupNotFoundException(parentGroupName);
            } else {
                throw new MembershipNotFoundException(childGroupName, parentGroupName);
            }
        } else if (Iterables.all(directoriesWithMembership, hasPermissions(application, UPDATE_GROUP_PERMISSION))) {
            for (Directory directory : directoriesWithMembership) {
                try {
                    directoryManager.removeGroupFromGroup(directory.getId(), childGroupName, parentGroupName);
                } catch (DirectoryPermissionException e) {
                    throw new ApplicationPermissionException(e);
                } catch (DirectoryNotFoundException e) {
                    // directory is no longer mapped, no need to remove the membership
                } catch (ReadOnlyGroupException e) {
                    throw new ApplicationPermissionException(e);
                } catch (InvalidMembershipException e) {
                    throw new OperationFailedException(
                            String.format("Cannot remove group %s from %s because they have different types",
                                    childGroupName, parentGroupName),
                            e);
                }
            }
        } else {
            throw new ApplicationPermissionException(String.format(
                    "At least one directory containing %s as a member of %s does not have write permission",
                    childGroupName, parentGroupName));
        }
    }

    @Override
    public boolean isUserDirectGroupMember(final Application application, final String username, final String groupName) {
        if (application.isMembershipAggregationEnabled()) {
            return Iterables.any(getActiveDirectories(application), containsUserDirectMembershipInGroup(username, groupName));
        } else {
            try {
                // check for memberships only in the directory where the canonical user lives
                User user = findUserByName(application, username);
                return directoryManager.isUserDirectGroupMember(user.getDirectoryId(), username, groupName);
            } catch (final UserNotFoundException e) {
                return false;
            } catch (final OperationFailedException e) {
                logger.error(e.getMessage(), e);
                return false; // if we cannot read from the directory, assume membership does not exist
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryIteration(e);
            }
        }
    }

    @Override
    public boolean isGroupDirectGroupMember(final Application application, final String childGroup, final String parentGroup) {
        if (application.isMembershipAggregationEnabled()) {
            return Iterables.any(getActiveDirectories(application),
                    containsGroupDirectMembershipInGroup(childGroup, parentGroup));
        } else {
            // look up only the directory where the canonical child group exists
            try {
                Group group = findGroupByName(application, childGroup);
                return directoryManager.isGroupDirectGroupMember(group.getDirectoryId(), childGroup, parentGroup);
            } catch (GroupNotFoundException e) {
                return false;
            } catch (OperationFailedException e) {
                logger.error(e.getMessage(), e);
                return false;  // if we cannot read from the directory, assume membership does not exist
            } catch (DirectoryNotFoundException e) {
                throw new ConcurrentModificationException("Directory mapping was removed while determining if the group is a direct group member: "
                        + e.getMessage());
            }
        }
    }

    @Override
    public boolean isUserNestedGroupMember(final Application application, final String username, final String groupName) {
        if (application.isMembershipAggregationEnabled()) {
            return Iterables.any(getActiveDirectories(application),
                    containsUserNestedMembershipInGroup(username, groupName));
        } else {
            try {
                User user = findUserByName(application, username);
                return directoryManager.isUserNestedGroupMember(user.getDirectoryId(), username, groupName);
            } catch (final UserNotFoundException e) {
                // do nothing
                return false;
            } catch (final OperationFailedException e) {
                logger.error(e.getMessage(), e);
                return false;  // if we cannot read from the directory, assume membership does not exist
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryAccess(e);
            }
        }
    }

    @Override
    public boolean isGroupNestedGroupMember(final Application application, final String childGroup, final String parentGroup) {
        if (application.isMembershipAggregationEnabled()) {
            return Iterables.any(getActiveDirectories(application),
                    containsGroupNestedMembershipInGroup(childGroup, parentGroup));
        } else {
            // look up only the directory where the canonical child group exists
            try {
                Group group = findGroupByName(application, childGroup);
                return directoryManager.isGroupNestedGroupMember(group.getDirectoryId(), childGroup, parentGroup);
            } catch (GroupNotFoundException e) {
                return false;
            } catch (OperationFailedException e) {
                logger.error(e.getMessage(), e);
                return false;  // if we cannot read from the directory, assume membership does not exist
            } catch (DirectoryNotFoundException e) {
                // Java 6 compatible constructor
                throw new ConcurrentModificationException("Directory mapping was removed while determining if the group is a nested group member: "
                        + e.getMessage());
            }
        }
    }

    @Override
    public <T> List<T> searchDirectGroupRelationships(final Application application, final MembershipQuery<T> query) {
        return getMembershipSearchStrategyOrFail(application).searchDirectGroupRelationships(query);
    }

    @Override
    public <T> List<T> searchNestedGroupRelationships(final Application application, final MembershipQuery<T> query) {
        return getMembershipSearchStrategyOrFail(application).searchNestedGroupRelationships(query);
    }

    /**
     * Given a username is duplicated in several {@link Directory user directories} under the same
     * {@link Application application}.
     *
     * <p>The user in the first directory, according to directory ordering is considered to
     * be the canonical user for the given username, and the other users are shadowed and thus not to be returned from
     * searches.</p>
     *
     * @param application performing the search
     * @param entity      to be determined as canonical or not
     * @return {@code true}; if the passed in user is canonical; otherwise, {@code false} is returned
     */
    <T extends DirectoryEntity> boolean isCanonical(final Application application, @Nullable T entity) {
        if (entity == null) {
            return true;
        }

        // Users in the first active directory cannot be shadowed
        Directory firstActive = Iterables.getFirst(getActiveDirectories(application), null);

        if (firstActive == null) {
            /* No directories; user doesn't exist */
            return false;
        } else if (firstActive.getId().equals(entity.getDirectoryId())) {
            return true;
        }

        try {
            final DirectoryEntity canonicalEntity;
            if (entity instanceof User) {
                canonicalEntity = findUserByName(application, entity.getName());
            } else if (entity instanceof Group) {
                canonicalEntity = findGroupByName(application, entity.getName());
            } else {
                throw new IllegalArgumentException("Entity must be an instance of User or Group (was " +
                        entity.getClass().getName() + ")");
            }
            return entity.getDirectoryId() == canonicalEntity.getDirectoryId();
        } catch (final UserNotFoundException | GroupNotFoundException e) {
            return false;
        }
    }

    @Override
    public String getCurrentEventToken(Application application) throws IncrementalSynchronisationNotAvailableException {
        final List<Directory> activeDirectories = ImmutableList.copyOf(getActiveDirectories(application));

        // We don't need to check this in getNewEvents, because all configuration changes will force this method to be
        // called first.
        assertIncrementalSynchronisationIsAvailable(activeDirectories);

        return eventStore.getCurrentEventToken(activeDirectories.stream().map(Directory::getId).collect(Collectors.toList()));
    }

    @Override
    public Events getNewEvents(Application application, String eventToken) throws EventTokenExpiredException, OperationFailedException {
        final Events events = eventStore.getNewEvents(eventToken, application);
        final List<OperationEvent> eventsList = ImmutableList.copyOf(events.getEvents());
        if ((application.isFilteringGroupsWithAccessEnabled() || application.isFilteringUsersWithAccessEnabled())
            && !eventsList.isEmpty()) {
            throw new EventTokenExpiredException("Incremental sync is not available when access based filtering is on.");
        }
        final List<OperationEvent> applicationEvents = new EventTransformer(directoryManager, application)
                .transformEvents(eventsList);

        return new Events(applicationEvents, events.getNewEventToken());
    }

    @Override
    public Webhook findWebhookById(Application application, long webhookId)
            throws WebhookNotFoundException, ApplicationPermissionException {
        Webhook webhook = webhookRegistry.findById(webhookId);
        if (application.getId().equals(webhook.getApplication().getId())) {
            return webhook;
        } else {
            throw new ApplicationPermissionException("Application does not own Webhook");
        }
    }

    @Override
    public Webhook registerWebhook(Application application, String endpointUrl, @Nullable String token) throws InvalidWebhookEndpointException {
        ensureWebhookEndpointUrlIsValid(endpointUrl);

        Webhook webhookTemplate = new WebhookTemplate(application, endpointUrl, token);
        return webhookRegistry.add(webhookTemplate);
    }

    @Override
    public void unregisterWebhook(Application application, long webhookId)
            throws ApplicationPermissionException, WebhookNotFoundException {
        Webhook webhook = webhookRegistry.findById(webhookId);
        if (application.getId().equals(webhook.getApplication().getId())) {
            webhookRegistry.remove(webhook);
        } else {
            throw new ApplicationPermissionException("Application does not own Webhook");
        }
    }

    @Override
    public UserCapabilities getCapabilitiesForNewUsers(final Application application) {
        Directory directory = findFirstDirectoryWithCreateUserPermission(application);
        if (directory == null) {
            return DirectoryUserCapabilities.none();
        } else {
            return DirectoryUserCapabilities.fromDirectory(directory);
        }
    }

    /**
     * Given an {@code application}, retrieve all active directories associated with it.
     *
     * @param application application to query
     * @return list of active directories associated with {@code application}
     * @since 2.7.2
     */
    protected List<Directory> getActiveDirectories(Application application) {
        return Applications.getActiveDirectories(application);
    }

    private MembershipSearchStrategy getMembershipSearchStrategyOrFail(Application application) {
        final List<Directory> activeDirectories = getActiveDirectories(application);
        return searchStrategyFactory.createMembershipSearchStrategy(
                application.isMembershipAggregationEnabled(), activeDirectories,
                new SimpleCanonicalityChecker(directoryManager, activeDirectories),
                simpleAccessFilter(application));
    }

    private GroupSearchStrategy getGroupSearchStrategyOrFail(Application application) {
        return searchStrategyFactory.createGroupSearchStrategy(true, getActiveDirectories(application),
                simpleAccessFilter(application));
    }

    private UserSearchStrategy getUserSearchStrategyOrFail(Application application) {
        return searchStrategyFactory.createUserSearchStrategy(true, getActiveDirectories(application),
                simpleAccessFilter(application));
    }

    private AccessFilter simpleAccessFilter(Application application) {
        return AccessFilters.create(directoryManager, application, false);
    }

    private static void ensureWebhookEndpointUrlIsValid(String endpointUrl) throws InvalidWebhookEndpointException {
        URI endpointUri;
        try {
            endpointUri = new URI(endpointUrl);
        } catch (URISyntaxException e) {
            throw new InvalidWebhookEndpointException(endpointUrl, e);
        }
        if (!endpointUri.isAbsolute()) {
            throw new InvalidWebhookEndpointException(endpointUrl, "because the url is not absolute");
        } else if (!"http".equalsIgnoreCase(endpointUri.getScheme())
                && !"https".equalsIgnoreCase(endpointUri.getScheme())) {
            throw new InvalidWebhookEndpointException(endpointUrl, "because the url scheme is not http or https");
        }
    }

    private void assertIncrementalSynchronisationIsAvailable(List<Directory> activeDirectories) throws IncrementalSynchronisationNotAvailableException {
        for (Directory directory : activeDirectories) {
            // No events are generated for cacheable directories that are not cached
            if (isFalse(toBooleanObject(directory.getValue(DirectoryProperties.CACHE_ENABLED)))) {
                throw new IncrementalSynchronisationNotAvailableException("Directory '" + directory.getName() + "' is not cached and so cannot be incrementally synchronised");
            }
        }
    }

    /**
     * Returns either CREATE_GROUP or CREATE_ROLE depending on the GroupType of the given Group
     *
     * @param group The Group
     * @return either CREATE_GROUP or CREATE_ROLE depending on the GroupType of the given Group
     */
    private OperationType getCreateOperationType(final Group group) {
        switch (group.getType()) {
            case GROUP:
                return OperationType.CREATE_GROUP;
            default:
                throw new UnsupportedOperationException();
        }
    }

    /**
     * Returns either UPDATE_GROUP or UPDATE_ROLE depending on the GroupType of the given Group
     *
     * @param group The Group
     * @return either UPDATE_GROUP or UPDATE_ROLE depending on the GroupType of the given Group
     */
    private OperationType getUpdateOperationType(final Group group) {
        switch (group.getType()) {
            case GROUP:
                return OperationType.UPDATE_GROUP;
            default:
                throw new UnsupportedOperationException();
        }
    }

    /**
     * Returns either UPDATE_GROUP_ATTRIBUTE or UPDATE_ROLE_ATTRIBUTE depending on the GroupType
     * of the given Group.
     *
     * @param group The Group
     * @return either UPDATE_GROUP_ATTRIBUTE or UPDATE_ROLE_ATTRIBUTE depending on the GroupType
     * of the given Group
     */
    private OperationType getUpdateAttributeOperationType(final Group group) {
        switch (group.getType()) {
            case GROUP:
                return OperationType.UPDATE_GROUP_ATTRIBUTE;
            default:
                throw new UnsupportedOperationException();
        }
    }

    /**
     * Returns either DELETE_GROUP or DELETE_ROLE depending on the GroupType of the given Group
     *
     * @param group The Group
     * @return either DELETE_GROUP or DELETE_ROLE depending on the GroupType of the given Group
     */
    private OperationType getDeleteOperationType(final Group group) {
        switch (group.getType()) {
            case GROUP:
                return OperationType.DELETE_GROUP;
            default:
                throw new UnsupportedOperationException();
        }
    }

    /**
     * Determines if a user is authorised to authenticate with a given application.
     * <p/>
     * For a user to have access to an application:
     * <ol>
     * <li>the Application must be active.</li>
     * <li>and either:
     * <ul>
     * <li>the User is stored in a directory which is associated to the Application and the "allow all to authenticate"
     * flag is true.</li>
     * <li>the User is a member of a Group that is allowed to authenticate with the Application and both the User and
     * Group are from the same RemoteDirectory.</li>
     * </ul></li>
     * </ol>
     * <p/>
     * Note that this call is not cached and does not affect the
     * cache.
     *
     * @param application application the user wants to authenticate with.
     * @param username    the username of the user that wants to authenticate with the application.
     * @param directoryId the directoryId of the user that wants to authenticate with the application.
     * @return <code>true</code> iff the user is authorised to authenticate with the application.
     * @throws OperationFailedException   if the directory implementation could not be loaded when performing a membership check.
     * @throws DirectoryNotFoundException
     */
    private boolean isAllowedToAuthenticate(final String username, final long directoryId, final Application application)
            throws OperationFailedException, DirectoryNotFoundException {
        // first make sure the application is not inactive
        if (!application.isActive()) {
            logger.debug("User does not have access to application '{}' as the application is inactive", application.getName());
            return false;
        }

        // see if the application has a mapping for the user's directory
        final ApplicationDirectoryMapping directoryMapping = application.getApplicationDirectoryMapping(directoryId);
        if (directoryMapping != null && (directoryMapping.isAllowAllToAuthenticate()
                || directoryManager.isUserNestedGroupMember(directoryId, username, directoryMapping.getAuthorisedGroupNames()))) {
            return true;
        }

        // did not have allow all on directory OR meet group membership requirements
        logger.debug("User does not have access to application '{}' as the directory is not allow all to authenticate and the user is not a member of any of the authorised groups", application.getName());
        return false;
    }

    private static ConcurrentModificationException concurrentModificationExceptionForDirectoryIteration(
            DirectoryNotFoundException e) {
        final ConcurrentModificationException concurrentModificationException = new ConcurrentModificationException(
                "Directory mapping was removed while iterating through directories");
        concurrentModificationException.initCause(e);
        return concurrentModificationException;
    }

    private static ConcurrentModificationException concurrentModificationExceptionForDirectoryAccess(
            DirectoryNotFoundException e) {
        final ConcurrentModificationException concurrentModificationException =
                new ConcurrentModificationException("Directory mapping was removed while accessing the directory");
        concurrentModificationException.initCause(e);
        return concurrentModificationException;
    }

    private Directory getDefiningDirectory(Application application, String username)
            throws OperationFailedException, UserNotFoundException {
        long dirId = fastFailingFindUser(application, username).getDirectoryId();
        return application.getApplicationDirectoryMapping(dirId).getDirectory();
    }

    @Override
    @Nullable
    public URI getUserAvatarLink(Application application, String username, int sizeHint)
            throws UserNotFoundException, DirectoryNotFoundException, OperationFailedException {
        Directory d = getDefiningDirectory(application, username);

        AvatarReference av = directoryManager.getUserAvatarByName(d.getId(), username, sizeHint);

        if (av instanceof AvatarReference.UriAvatarReference) {
            return ((AvatarReference.UriAvatarReference) av).getUri();
        } else if (av instanceof AvatarReference.BlobAvatar) {
            /* If the avatar is a blob, we can't serve it to the application. Pass back a link
             * so that the user will then request it directly.
             */
            return avatarProvider.getHostedUserAvatarUrl(application.getId(), username, sizeHint);
        } else {
            /* Use a generic avatar derived from the user's details, if any */
            User user = directoryManager.findUserByName(d.getId(), username);
            return avatarProvider.getUserAvatar(user, sizeHint);
        }
    }

    @Override
    @Nullable
    public AvatarReference getUserAvatar(Application application, String username, int sizeHint)
            throws UserNotFoundException, DirectoryNotFoundException, OperationFailedException {
        Directory d = getDefiningDirectory(application, username);

        AvatarReference av = directoryManager.getUserAvatarByName(d.getId(), username, sizeHint);

        if (av != null) {
            return av;
        } else {
            User user = directoryManager.findUserByName(d.getId(), username);
            URI uri = avatarProvider.getUserAvatar(user, sizeHint);
            if (uri != null) {
                return new AvatarReference.UriAvatarReference(uri);
            } else {
                return null;
            }
        }
    }

    @Override
    public void expireAllPasswords(Application application) throws OperationFailedException {
        logger.info("Expiring all passwords for application {}", application.getName());
        for (Directory directory : getActiveDirectories(application)) {
            try {
                if (directoryManager.supportsExpireAllPasswords(directory.getId())) {
                    directoryManager.expireAllPasswords(directory.getId());
                }
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(e);
            }
        }
    }

    @Override
    public User userAuthenticated(Application application, String username) throws UserNotFoundException, OperationFailedException, InactiveAccountException {
        OperationFailedException initialException = null;
        List<Directory> sortedDirectories = authenticationOrderOptimizer.optimizeDirectoryOrderForAuthentication(application, getActiveDirectories(application), username);

        for (final Directory directory : sortedDirectories) {
            try {
                final User user = directoryManager.userAuthenticated(directory.getId(), username);
                eventPublisher.publish(new UserAuthenticatedEvent(this, directory, application, user));
                return user;
            } catch (OperationFailedException e) {
                logger.debug("userAuthenticated() failed for user {} directory {}, continuing", username, directory.getId(), e);
                if (initialException == null) {
                    initialException = e;
                }
            } catch (UserNotFoundException e) {
                logger.debug("User not found during userAuthenticated() for user {} directory {}, continuing", username, directory.getId(), e);
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryIteration(e);
            }
        }

        if (initialException != null) {
            throw initialException;
        }

        throw new UserNotFoundException(username);
    }

    @Override
    public MembershipsIterable getMemberships(Application application) {
        return MembershipsIterableImpl.runWithClassLoader(Thread.currentThread().getContextClassLoader(),
                new MembershipsIterableImpl(directoryManager, searchStrategyFactory, application));
    }
}
