package com.atlassian.crowd.manager.application;

import com.atlassian.crowd.dao.application.ApplicationDAO;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.OperationType;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.event.application.ApplicationCreatedEvent;
import com.atlassian.crowd.event.application.ApplicationDeletedEvent;
import com.atlassian.crowd.event.application.ApplicationDirectoryAddedEvent;
import com.atlassian.crowd.event.application.ApplicationDirectoryOrderUpdatedEvent;
import com.atlassian.crowd.event.application.ApplicationDirectoryRemovedEvent;
import com.atlassian.crowd.event.application.ApplicationRemoteAddressAddedEvent;
import com.atlassian.crowd.event.application.ApplicationRemoteAddressRemovedEvent;
import com.atlassian.crowd.event.application.ApplicationUpdatedEvent;
import com.atlassian.crowd.exception.ApplicationAlreadyExistsException;
import com.atlassian.crowd.exception.ApplicationNotFoundException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;
import com.atlassian.crowd.model.application.ApplicationType;
import com.atlassian.crowd.model.application.ImmutableApplication;
import com.atlassian.crowd.model.application.RemoteAddress;
import com.atlassian.crowd.password.encoder.PasswordEncoder;
import com.atlassian.crowd.password.encoder.UpgradeablePasswordEncoder;
import com.atlassian.crowd.password.factory.PasswordEncoderFactory;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.constants.ApplicationTermKeys;
import com.atlassian.event.api.EventPublisher;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Set;

import static org.apache.commons.lang3.Validate.notNull;

@Transactional
public class ApplicationManagerGeneric implements ApplicationManager {
    private static final Logger logger = LoggerFactory.getLogger(ApplicationManagerGeneric.class);

    private final ApplicationDAO applicationDao;
    private final PasswordEncoderFactory passwordEncoderFactory;
    private final EventPublisher eventPublisher;

    public ApplicationManagerGeneric(ApplicationDAO applicationDao,
                                     PasswordEncoderFactory passwordEncoderFactory,
                                     EventPublisher eventPublisher) {
        this.applicationDao = applicationDao;
        this.passwordEncoderFactory = passwordEncoderFactory;
        this.eventPublisher = eventPublisher;
    }

    public Application add(Application application) throws InvalidCredentialException, ApplicationAlreadyExistsException {
        Validate.notNull(application, "application should not be null");
        if (application.getCredential() == null) {
            throw new InvalidCredentialException("Password of the application cannot be null");
        }
        if (applicationWithNameExists(application.getName())) {
            throw new ApplicationAlreadyExistsException("An application with the specified name already exists");
        }
        PasswordCredential encryptedCredential = encryptAndUpdateApplicationCredential(application.getCredential());
        final Application app = applicationDao.add(application, encryptedCredential);
        eventPublisher.publish(new ApplicationCreatedEvent(app));
        return app;
    }

    public Application findById(long id) throws ApplicationNotFoundException {
        return applicationDao.findById(id);
    }

    public Application findByName(String name) throws ApplicationNotFoundException {
        return applicationDao.findByName(name);
    }

    public void remove(Application application) throws ApplicationManagerException {
        if (application.isPermanent()) {
            throw new ApplicationManagerException("Cannot delete a permanent application");
        }

        // remove the application
        try {
            final Application oldApplication = ImmutableApplication.from(findById(application.getId()));
            applicationDao.remove(application);
            eventPublisher.publish(new ApplicationDeletedEvent(oldApplication));
        } catch (ApplicationNotFoundException e) {
            throw new IllegalArgumentException(e);
        }
    }

    public void removeDirectoryFromApplication(Directory directory, Application application) throws ApplicationManagerException {
        ApplicationDirectoryMapping mapping = application.getApplicationDirectoryMapping(directory.getId());
        if (mapping != null) {
            try {
                final Application oldApplication = ImmutableApplication.from(findById(application.getId()));
                applicationDao.removeDirectoryMapping(application.getId(), directory.getId());
                final Application newApplication = ImmutableApplication.from(findById(application.getId()));
                eventPublisher.publish(new ApplicationDirectoryRemovedEvent(oldApplication, newApplication, directory));
            } catch (ApplicationNotFoundException e) {
                // do nothing since we wanted to delete the directory anyway
            }
        }
    }

    public List<Application> search(EntityQuery query) {
        return applicationDao.search(query);
    }

    public List<Application> findAll() {
        return search(QueryBuilder.queryFor(Application.class, EntityDescriptor.application()).returningAtMost(EntityQuery.ALL_RESULTS));
    }

    public Application update(Application application) throws ApplicationManagerException, ApplicationNotFoundException {
        // cannot deactivate crowd
        if (application.getType() == ApplicationType.CROWD && !application.isActive()) {
            throw new ApplicationManagerException("Cannot deactivate the Crowd application");
        }

        // cannot rename permanent applications
        if (application.isPermanent()) {
            try {
                Application savedApp = findById(application.getId());
                if (!savedApp.getName().equals(application.getName())) {
                    throw new ApplicationManagerException("Cannot rename a permanent application");
                }
            } catch (ApplicationNotFoundException e) {
                throw new ApplicationManagerException(e.getMessage(), e);
            }
        }
        // Application names cannot be changed to the name of a current Crowd Application.
        Application currentApplication;
        try {
            currentApplication = findByName(application.getName());
        } catch (ApplicationNotFoundException e) {
            // We are changing the name of the application to one that doesn't exist, this is OK.
            // Just get the current application by Id, so we can make sure we are actually trying to update the correct application.
            currentApplication = findById(application.getId());
        }

        // If we are updating an application, make sure that it is the passed in Application
        if (application.getId().equals(currentApplication.getId())) {
            final Application oldApplication = ImmutableApplication.from(currentApplication);
            Application savedApplication = applicationDao.update(application);
            eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, ImmutableApplication.from(findById(oldApplication.getId()))));
            return savedApplication;
        }

        throw new ApplicationManagerException("You potentially tried to update an application with a different ID than the one you passed in");
    }

    public void updateCredential(Application application, PasswordCredential passwordCredential)
            throws ApplicationManagerException, ApplicationNotFoundException {
        notNull(application);
        notNull(passwordCredential);
        notNull(passwordCredential.getCredential());

        final PasswordCredential oldPassword = new PasswordCredential(application.getCredential());
        final Application oldApplication = ImmutableApplication.builder(findById(application.getId())).setPasswordCredential(oldPassword).build();
        PasswordCredential encryptedCredential = encryptAndUpdateApplicationCredential(passwordCredential);
        applicationDao.updateCredential(application, encryptedCredential);
        eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, ImmutableApplication.from(findById(application.getId()))));
    }

    public boolean authenticate(Application application, PasswordCredential testCredential)
            throws ApplicationNotFoundException {
        notNull(application);
        notNull(testCredential);
        notNull(testCredential.getCredential());

        if (PasswordCredential.NONE.equals(application.getCredential())) {
            return false;
        }

        final PasswordEncoder encoder = getAtlassianSecurityEncoder();
        if (!encoder.isPasswordValid(application.getCredential().getCredential(), testCredential.getCredential(), null)) {
            return false;
        }

        upgradePasswordIfRequired(application, encoder, testCredential.getCredential());

        return true;
    }

    private void upgradePasswordIfRequired(Application application, PasswordEncoder encoder, String rawPass)
            throws ApplicationNotFoundException {
        // When using UpgradeablePasswordEncoder, we might be asked to re-encode the password.
        if (encoder instanceof UpgradeablePasswordEncoder) {
            final UpgradeablePasswordEncoder upgradeableEncoder = (UpgradeablePasswordEncoder) encoder;
            if (upgradeableEncoder.isUpgradeRequired(application.getCredential().getCredential())) {
                final String newEncPass = encoder.encodePassword(rawPass, null);
                applicationDao.updateCredential(application, new PasswordCredential(newEncPass, true));
            }
        }
    }

    public void addDirectoryMapping(Application application, Directory directory, boolean allowAllToAuthenticate, OperationType... operationTypes)
            throws ApplicationNotFoundException, DirectoryNotFoundException {
        notNull(application);
        notNull(application.getId());
        notNull(directory);
        notNull(directory.getId());

        final Application oldApplication = ImmutableApplication.from(findById(application.getId()));
        applicationDao.addDirectoryMapping(application.getId(), directory.getId(), allowAllToAuthenticate, operationTypes);
        final Application newApplication = ImmutableApplication.from(findById(application.getId()));
        eventPublisher.publish(new ApplicationDirectoryAddedEvent(oldApplication, newApplication, directory));
    }

    public void updateDirectoryMapping(Application application, Directory directory, int position)
            throws ApplicationNotFoundException, DirectoryNotFoundException {
        notNull(application);
        notNull(application.getId());
        notNull(directory);
        notNull(directory.getId());

        final Application oldApplication = ImmutableApplication.from(findById(application.getId()));
        applicationDao.updateDirectoryMapping(application.getId(), directory.getId(), position);
        final Application newApplication = ImmutableApplication.from(findById(application.getId()));

        eventPublisher.publish(new ApplicationDirectoryOrderUpdatedEvent(oldApplication, newApplication, directory));
        logger.debug("Changed directory mapping order within application <{}> of directory <{}> to position <{}>",
                application.getId(), directory.getId(), position);
    }

    public void updateDirectoryMapping(Application application, Directory directory, boolean allowAllToAuthenticate)
            throws ApplicationNotFoundException, DirectoryNotFoundException {
        notNull(application);
        notNull(application.getId());
        notNull(directory);
        notNull(directory.getId());

        final Application oldApplication = ImmutableApplication.from(findById(application.getId()));

        applicationDao.updateDirectoryMapping(application.getId(), directory.getId(), allowAllToAuthenticate);
        final Application newApplication = ImmutableApplication.from(findById(application.getId()));

        eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, newApplication));
    }

    public void updateDirectoryMapping(Application application, Directory directory, boolean allowAllToAuthenticate, Set<OperationType> operationTypes)
            throws ApplicationNotFoundException, DirectoryNotFoundException {
        notNull(application);
        notNull(application.getId());
        notNull(directory);
        notNull(directory.getId());

        final Application oldApplication = ImmutableApplication.from(findById(application.getId()));

        applicationDao.updateDirectoryMapping(application.getId(), directory.getId(), allowAllToAuthenticate, operationTypes);

        final Application newApplication = ImmutableApplication.from(findById(application.getId()));

        eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, newApplication));
    }

    public void addRemoteAddress(Application application, RemoteAddress remoteAddress) throws ApplicationNotFoundException {
        notNull(application);
        notNull(application.getId());

        final Application oldApplication = ImmutableApplication.from(findById(application.getId()));

        applicationDao.addRemoteAddress(application.getId(), remoteAddress);
        final Application newApplication = ImmutableApplication.from(findById(application.getId()));
        eventPublisher.publish(new ApplicationRemoteAddressAddedEvent(application, remoteAddress));

        eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, newApplication));
    }

    public void removeRemoteAddress(Application application, RemoteAddress remoteAddress) throws ApplicationNotFoundException {
        notNull(application);
        notNull(application.getId());
        notNull(remoteAddress);
        final Application oldApplication = ImmutableApplication.from(findById(application.getId()));
        applicationDao.removeRemoteAddress(application.getId(), remoteAddress);
        eventPublisher.publish(new ApplicationRemoteAddressRemovedEvent(application, remoteAddress));
        eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, application));
    }

    public void addGroupMapping(Application application, Directory directory, String groupName)
            throws ApplicationNotFoundException {
        notNull(application);
        notNull(application.getId());
        notNull(directory);
        notNull(directory.getId());

        final Application oldApplication = ImmutableApplication.from(findById(application.getId()));

        applicationDao.addGroupMapping(application.getId(), directory.getId(), groupName);

        final Application newApplication = ImmutableApplication.from(findById(application.getId()));

        eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, newApplication));
    }

    public void removeGroupMapping(Application application, Directory directory, String groupName)
            throws ApplicationNotFoundException {
        notNull(application);
        notNull(application.getId());
        notNull(directory);
        notNull(directory.getId());

        final Application oldApplication = ImmutableApplication.from(findById(application.getId()));

        applicationDao.removeGroupMapping(application.getId(), directory.getId(), groupName);

        final Application newApplication = ImmutableApplication.from(findById(application.getId()));

        eventPublisher.publish(new ApplicationUpdatedEvent(oldApplication, newApplication));
    }

    private PasswordCredential encryptAndUpdateApplicationCredential(PasswordCredential passwordCredential) {
        PasswordEncoder encoder = getAtlassianSecurityEncoder();

        String encryptedPassword = encoder.encodePassword(passwordCredential.getCredential(), null);

        return new PasswordCredential(encryptedPassword, true);
    }

    private PasswordEncoder getAtlassianSecurityEncoder() {
        return passwordEncoderFactory.getEncoder(PasswordEncoderFactory.ATLASSIAN_SECURITY_ENCODER);
    }

    private boolean applicationWithNameExists(String name) {
        final EntityQuery<Application> query = QueryBuilder
                .queryFor(Application.class, EntityDescriptor.application())
                .with(Restriction.on(ApplicationTermKeys.NAME).exactlyMatching(name))
                .returningAtMost(1);
        final List<Application> results = applicationDao.search(query);
        return results.size() > 0;
    }
}
