package com.atlassian.crowd.lock;

import com.atlassian.beehive.ClusterLock;

import java.util.function.Supplier;

/**
 * Helper class supporting lambdas for {@link ClusterLock}
 */
public class ClusterLockWrapper {
    private final Supplier<ClusterLock> lockSupplier;

    /**
     * @param lockSupplier supplier of the lock, under which all operations will be performed
     */
    public ClusterLockWrapper(Supplier<ClusterLock> lockSupplier) {
        this.lockSupplier = lockSupplier;
    }

    /**
     * Runs provided {@link Runnable} under the lock. Blocks to get the lock.
     * @param runnable {@link Runnable} to be run under the lock
     */
    public void run(Runnable runnable) {
        ClusterLock lock = lockSupplier.get();
        lock.lock();
        try {
            runnable.run();
        } finally {
            lock.unlock();
        }
    }

    /**
     * Runs provided {@link Supplier} under the lock. Blocks to get the lock.
     * @param supplier {@link Supplier} to be run under the lock
     * @return value returned by the supplier
     */
    public <T> T run(Supplier<T> supplier) {
        ClusterLock lock = lockSupplier.get();
        lock.lock();
        try {
            return supplier.get();
        } finally {
            lock.unlock();
        }
    }

    /**
     * Tries to get the lock and runs provided {@link Runnable} under the lock. If the lock wasn't available
     * immediately then does nothing and returns false.
     * @param runnable {@link Runnable} to be run under the lock
     * @return whether the lock was acquired and {@link Runnable} was run
     */
    public boolean tryRun(Runnable runnable) {
        ClusterLock lock = lockSupplier.get();
        if (lock.tryLock()) {
            try {
                runnable.run();
                return true;
            } finally {
                lock.unlock();
            }
        } else {
            return false;
        }
    }
}
