package com.atlassian.crowd.event;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.event.application.ApplicationUpdatedEvent;
import com.atlassian.crowd.event.directory.DirectoryDeletedEvent;
import com.atlassian.crowd.event.directory.DirectoryUpdatedEvent;
import com.atlassian.crowd.event.group.GroupAttributeDeletedEvent;
import com.atlassian.crowd.event.group.GroupAttributeStoredEvent;
import com.atlassian.crowd.event.group.GroupCreatedEvent;
import com.atlassian.crowd.event.group.GroupDeletedEvent;
import com.atlassian.crowd.event.group.GroupMembershipDeletedEvent;
import com.atlassian.crowd.event.group.GroupMembershipsCreatedEvent;
import com.atlassian.crowd.event.group.GroupUpdatedEvent;
import com.atlassian.crowd.event.migration.XMLRestoreFinishedEvent;
import com.atlassian.crowd.event.user.UserAttributeDeletedEvent;
import com.atlassian.crowd.event.user.UserAttributeStoredEvent;
import com.atlassian.crowd.event.user.UserCreatedEvent;
import com.atlassian.crowd.event.user.UserRenamedEvent;
import com.atlassian.crowd.event.user.UserUpdatedEvent;
import com.atlassian.crowd.event.user.UsersDeletedEvent;
import com.atlassian.crowd.manager.webhook.WebhookService;
import com.atlassian.crowd.model.event.AliasEvent;
import com.atlassian.crowd.model.event.GroupEvent;
import com.atlassian.crowd.model.event.GroupMembershipEvent;
import com.atlassian.crowd.model.event.Operation;
import com.atlassian.crowd.model.event.OperationEvent;
import com.atlassian.crowd.model.event.UserEvent;
import com.atlassian.crowd.model.event.UserMembershipEvent;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.ImmutableGroup;
import com.atlassian.crowd.model.membership.MembershipType;
import com.atlassian.crowd.model.user.ImmutableUser;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;

import static com.atlassian.crowd.attribute.AttributePredicates.SYNCING_ATTRIBUTE;
import static com.atlassian.crowd.model.membership.MembershipType.GROUP_USER;

/**
 * This class listens for events related to user and group changes and saves
 * them to {@link EventStore}. It also notifies {@link com.atlassian.crowd.manager.webhook.WebhookService} when
 * new information is available in the EventStore.
 */
public class StoringEventListener {
    private final EventStore eventStore;
    private final WebhookService webhookService;

    public StoringEventListener(final EventStore eventStore, final EventPublisher eventPublisher,
                                final WebhookService webhookService) {
        this.eventStore = eventStore;
        this.webhookService = webhookService;
        eventPublisher.register(this);
    }

    private void storeEventAndNotifyWebhooks(OperationEvent event) {
        eventStore.storeOperationEvent(event);
        webhookService.notifyWebhooks();
    }

    private void storeEventsAndNotifyWebhooks(Stream<OperationEvent> events) {
        events.forEach(event -> eventStore.storeOperationEvent(event));
        webhookService.notifyWebhooks();
    }

    private void invalidateEventsAndNotifyWebhooks(Object event) {
        eventStore.handleApplicationEvent(event);
        webhookService.notifyWebhooks();
    }

    @EventListener
    public void handleEvent(UserCreatedEvent event) {
        storeEventAndNotifyWebhooks(new UserEvent(Operation.CREATED, event.getDirectoryId(), event.getUser(), null, null));
    }

    @EventListener
    public void handleEvent(UserUpdatedEvent event) {
        Map<String, Set<String>> storedAttributes = null;
        Set<String> deletedAttributes = null;

        if (event instanceof UserAttributeStoredEvent) {
            final UserAttributeStoredEvent userAttributeStoredEvent = (UserAttributeStoredEvent) event;
            storedAttributes = userAttributeStoredEvent.getAttributeNames().stream()
                    .filter(SYNCING_ATTRIBUTE)
                    .collect(Collectors.toMap(attrName -> attrName, userAttributeStoredEvent::getAttributeValues));
            if (storedAttributes.isEmpty()) {
                return;
            }
        } else if (event instanceof UserAttributeDeletedEvent) {
            String attributeName = ((UserAttributeDeletedEvent) event).getAttributeName();
            if (!SYNCING_ATTRIBUTE.test(attributeName)) {
                return;
            }
            deletedAttributes = Collections.singleton(attributeName);
        } else if (event instanceof UserRenamedEvent) {
            invalidateEventsAndNotifyWebhooks(event);
            return;
        }

        storeEventAndNotifyWebhooks(new UserEvent(Operation.UPDATED,
                event.getDirectoryId(),
                event.getUser(),
                storedAttributes,
                deletedAttributes));
    }

    @EventListener
    public void handleEvent(UsersDeletedEvent event) {
        final Directory directory = event.getDirectory();
        final Long directoryId = directory.getId();
        storeEventsAndNotifyWebhooks(event.getUsernames().stream()
                .map(username -> new UserEvent(Operation.DELETED, directoryId, new ImmutableUser(new UserTemplate(username, directoryId)), null, null)));
    }

    @EventListener
    public void handleEvent(GroupCreatedEvent event) {
        storeEventAndNotifyWebhooks(new GroupEvent(Operation.CREATED,
                event.getDirectory().getId(),
                event.getGroup(),
                null,
                null));
    }

    @EventListener
    public void handleEvent(GroupUpdatedEvent event) {
        Map<String, Set<String>> storedAttributes = null;
        Set<String> deletedAttributes = null;

        if (event instanceof GroupAttributeStoredEvent) {
            final GroupAttributeStoredEvent groupAttributeStoredEvent = (GroupAttributeStoredEvent) event;
            storedAttributes = groupAttributeStoredEvent.getAttributeNames().stream()
                    .filter(SYNCING_ATTRIBUTE)
                    .collect(Collectors.toMap(attrName -> attrName, groupAttributeStoredEvent::getAttributeValues));
            if (storedAttributes.isEmpty()) {
                return;
            }
        } else if (event instanceof GroupAttributeDeletedEvent) {
            String attributeName = ((GroupAttributeDeletedEvent) event).getAttributeName();
            if (!SYNCING_ATTRIBUTE.test(attributeName)) {
                return;
            }
            deletedAttributes = Collections.singleton(attributeName);
        }

        storeEventAndNotifyWebhooks(new GroupEvent(Operation.UPDATED, event.getDirectory().getId(), event.getGroup(),
                storedAttributes, deletedAttributes));
    }

    @EventListener
    public void handleEvent(GroupDeletedEvent event) {
        final Group group = ImmutableGroup.builder(event.getDirectoryId(), event.getGroupName()).build();
        storeEventAndNotifyWebhooks(new GroupEvent(Operation.DELETED, event.getDirectory().getId(), group, null, null));
    }

    @EventListener
    public void handleEvent(GroupMembershipsCreatedEvent event) {
        final Stream<OperationEvent> eventStream;
        switch (event.getMembershipType()) {
            case GROUP_USER:
                eventStream = event.getEntityNames().stream().map(entityName -> new UserMembershipEvent(Operation.CREATED, event.getDirectory().getId(), entityName, event.getGroupName()));
                break;
            case GROUP_GROUP:
                eventStream = event.getEntityNames().stream().map(entityName -> new GroupMembershipEvent(Operation.CREATED, event.getDirectory().getId(), entityName, event.getGroupName()));
                break;
            default:
                throw new IllegalArgumentException("MembershipType " + event.getMembershipType() + " is not supported");
        }

        eventStream.forEach(eventStore::storeOperationEvent);
        webhookService.notifyWebhooks();
    }

    @EventListener
    public void handleEvent(GroupMembershipDeletedEvent event) {
        if (event.getMembershipType() == GROUP_USER) {
            storeEventAndNotifyWebhooks(new UserMembershipEvent(Operation.DELETED, event.getDirectoryId(),
                    event.getEntityName(), event.getGroupName()));
        } else if (event.getMembershipType() == MembershipType.GROUP_GROUP) {
            storeEventAndNotifyWebhooks(new GroupMembershipEvent(Operation.DELETED, event.getDirectoryId(),
                    event.getEntityName(), event.getGroupName()));
        } else {
            throw new IllegalArgumentException("MembershipType " + event.getMembershipType() + " is not supported");
        }
    }

    @EventListener
    public void handleEvent(AliasEvent event) {
        eventStore.storeOperationEvent(event);
    }

    // Any event that causes major or unknown changes to the user and group
    // data seen by the application should force application to do a full sync.

    @EventListener
    public void handleEvent(DirectoryUpdatedEvent event) {
        invalidateEventsAndNotifyWebhooks(event);
    }

    @EventListener
    public void handleEvent(DirectoryDeletedEvent event) {
        invalidateEventsAndNotifyWebhooks(event);
    }

    @EventListener
    public void handleEvent(XMLRestoreFinishedEvent event) {
        invalidateEventsAndNotifyWebhooks(event);
    }

    @EventListener
    public void handleEvent(ApplicationUpdatedEvent event) {
        invalidateEventsAndNotifyWebhooks(event);
    }
}