package com.atlassian.crowd.event;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.stream.Collectors;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.Applications;
import com.atlassian.crowd.model.event.Operation;
import com.atlassian.crowd.model.event.OperationEvent;

/**
 * Thread-safe {@link EventStore} implementation that uses main memory as a
 * backing store. The amount of events stored is fixed, and after reaching the
 * limit, the oldest events will start to get dropped.
 */
public final class EventStoreGeneric implements EventStore {
    private final static String SEPARATOR = ":";
    private final static Random random = new Random();

    private final long maxStoredEvents;
    private final ConcurrentNavigableMap<Long, OperationEvent> events = new ConcurrentSkipListMap<Long, OperationEvent>();
    private final String instanceId = String.valueOf(random.nextLong());

    private long eventNumber = 0;

    /**
     * Creates a new EventStoreGeneric instance.
     *
     * @param maxStoredEvents maximum amount of events to keep in the memory
     */
    public EventStoreGeneric(long maxStoredEvents) {
        this.maxStoredEvents = maxStoredEvents + 1;
        storeOperationEvent(new ResetEvent());
    }

    public String getCurrentEventToken(List<Long> directoryIds) {
        return toEventToken(events.lastKey());
    }

    @Override
    public Events getNewEvents(String eventToken, List<Long> directoryIds) throws EventTokenExpiredException {
        final Long currentEventNumber = toEventNumber(eventToken);
        final Iterator<Map.Entry<Long, OperationEvent>> eventsSince = events.tailMap(currentEventNumber).entrySet().iterator();

        if (!eventsSince.hasNext() || !eventsSince.next().getKey().equals(currentEventNumber)) {
            throw new EventTokenExpiredException();
        }

        final List<OperationEvent> events = new ArrayList<OperationEvent>();
        Long newEventNumber = currentEventNumber;
        while (eventsSince.hasNext()) {
            final Map.Entry<Long, OperationEvent> eventEntry = eventsSince.next();
            final OperationEvent event = eventEntry.getValue();
            if (event instanceof ResetEvent) {
                throw new EventTokenExpiredException(((ResetEvent) event).getResetReason());
            }
            newEventNumber = eventEntry.getKey();
            if (event.getDirectoryId() == null || directoryIds.contains(event.getDirectoryId())) {
                events.add(event);
            }
        }
        return new Events(events, toEventToken(newEventNumber));
    }

    @Override
    public Events getNewEvents(String eventToken, Application application) throws EventTokenExpiredException {
        return getNewEvents(eventToken, Applications.getActiveDirectories(application).stream().map(Directory::getId).collect(Collectors.toList()));
    }

    public synchronized void storeOperationEvent(OperationEvent event) {
        final long currentEventNumber = ++eventNumber;

        // Keep only fixed amount of events
        if (currentEventNumber > maxStoredEvents) {
            events.remove(events.firstKey());
        }

        events.put(currentEventNumber, event);
    }

    @Override
    public void handleApplicationEvent(Object event) {
        storeOperationEvent(ResetEvent.fromUnsupportedEvent(event.getClass()));
    }

    private Long toEventNumber(String eventToken) throws EventTokenExpiredException {
        final String[] parts = eventToken.split(SEPARATOR);
        if (parts.length != 2 || !parts[0].equals(instanceId)) {
            throw new EventTokenExpiredException();
        }
        return Long.valueOf(parts[1]);
    }

    private String toEventToken(long eventNumber) {
        return instanceId + SEPARATOR + eventNumber;
    }

    private static class ResetEvent implements OperationEvent {
        private final String resetReason;

        public ResetEvent() {
            this.resetReason = null;
        }

        public ResetEvent(String resetReason) {
            this.resetReason = resetReason;
        }

        public static ResetEvent fromUnsupportedEvent(Class unsupportedEvent) {
            return ResetEvent.withReason(unsupportedEvent.getName() + " is not supported by incremental sync.");
        }

        public static ResetEvent withReason(String reason) {
            return new ResetEvent(reason);
        }

        public String getResetReason() {
            return this.resetReason;
        }

        @Override
        public Operation getOperation() {
            return null;
        }

        @Override
        public Long getDirectoryId() {
            return null;
        }
    }

}
