package com.atlassian.crowd.directory.synchronisation.cache;

import com.atlassian.crowd.directory.DirectoryCacheChangeOperations;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.InternalDirectoryGroup;

import javax.annotation.Nullable;
import java.util.Date;

import static com.atlassian.crowd.util.EqualityUtil.different;

/**
 * A group action strategy that compares the remote group with a local group matched by external id. This adds support
 * for renaming groups. As this class is unaware of other groups in the cache/from the remote, care must be taken to
 * ensure no duplicates exist beforehand.
 *
 * This implementation also skips some checks from {@link DefaultGroupActionStrategy}, mostly related to groups of type
 * {@link com.atlassian.crowd.model.group.GroupType#LEGACY_ROLE}, so it is not intended as a drop in replacement for
 * existing LDAP directories, even if they support the notion of external ids for groups.
 */
public class ExternalIdCheckingGroupActionStrategy extends AbstractGroupActionStrategy {

    @Override
    public DirectoryCacheChangeOperations.GroupsToAddUpdateReplace decide(
            @Nullable InternalDirectoryGroup internalGroup,
            @Nullable InternalDirectoryGroup groupMatchedByExternalId,
            Group remoteGroup,
            Date syncStartDate,
            long directoryId) throws OperationFailedException {
        if (internalGroup != null && internalGroup.isLocal()) {
            return NO_OP;
        }

        if (groupMatchedByExternalId == null) {
            return addGroup(remoteGroup);
        }

        if (wasGroupUpdatedAfterSearchStart(remoteGroup, groupMatchedByExternalId, syncStartDate, directoryId)) {
            return NO_OP;
        }

        if (different(groupMatchedByExternalId.getName(), remoteGroup.getName())) {
            return replaceGroup(groupMatchedByExternalId, remoteGroup);
        }

        if (different(groupMatchedByExternalId.getDescription(), remoteGroup.getDescription())) {
            return updateGroup(groupMatchedByExternalId, remoteGroup);
        }

        return NO_OP;
    }
}
