package com.atlassian.crowd.directory.loader;

import java.util.Map;
import java.util.concurrent.ExecutionException;

import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.util.InstanceFactory;

import com.google.common.base.Function;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * CustomDirectoryInstanceLoader loads a RemoteDirectory by using an {@link com.atlassian.crowd.util.InstanceFactory}
 * to create a RemoteDirectory.
 *
 * @since v2.1
 */
public class CustomDirectoryInstanceLoader extends AbstractDirectoryInstanceLoader implements DirectoryInstanceLoader {
    private static final Logger logger = LoggerFactory.getLogger(CustomDirectoryInstanceLoader.class);
    private final LoadingCache<String, Boolean> canLoadCache;
    private final InstanceFactory instanceFactory;

    public CustomDirectoryInstanceLoader(final InstanceFactory instanceFactory) {
        this.instanceFactory = checkNotNull(instanceFactory);

        // store the result of attempting to load the specified RemoteDirectory class in a cache. This cache does
        // not need to be cluster coherent and as it is merely memoizing a 'pure function'
        canLoadCache = CacheBuilder.newBuilder().weakKeys().build(CacheLoader.from(new Function<String, Boolean>() {
            public Boolean apply(final String className) {
                try {
                    final Class<?> clazz = CustomDirectoryInstanceLoader.this.getClass().getClassLoader().loadClass(className);
                    return RemoteDirectory.class.isAssignableFrom(clazz);
                } catch (ClassNotFoundException e) {
                    logger.warn("Could not load class: {}", className);
                    return Boolean.FALSE;
                }
            }
        }));
    }

    public RemoteDirectory getRawDirectory(final Long id, final String className, final Map<String, String> attributes)
            throws DirectoryInstantiationException {
        return RemoteDirectoryInstanceFactoryUtil.newRemoteDirectory(instanceFactory, id, className, attributes);
    }

    public boolean canLoad(final String className) {
        try {
            return canLoadCache.get(className); // can do this only because we are using a LoadingCache
        } catch (ExecutionException e) {
            logger.warn("Failed to check class: {}", className, e);
            return Boolean.FALSE;
        }
    }
}
