package com.atlassian.crowd.directory.loader;

import com.atlassian.crowd.directory.AuditingDirectoryDecorator;
import com.atlassian.crowd.directory.DirectoryProperties;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.ldap.cache.CacheRefresherFactory;
import com.atlassian.crowd.directory.ldap.cache.DirectoryCacheFactory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.manager.audit.AuditService;
import com.atlassian.crowd.manager.audit.mapper.AuditLogGroupMapper;
import com.atlassian.crowd.manager.audit.mapper.AuditLogUserMapper;
import com.atlassian.event.api.EventPublisher;

/**
 * A subclass of {@link DbCachingRemoteDirectoryInstanceLoader} which allows DB caching to be switched off based
 * on the {@link DirectoryProperties#CACHE_ENABLED} attribute. Since 3.2.0 the uncached directories returned by this
 * class will be extended with auditing capabilities.
 *
 * <p>
 * Note, the caching in this class refers to the DB caching of directory data, not of in memory caches of the
 * {@link RemoteDirectory} instances themselves
 *
 * @since 2.8
 */
public class ConfigurableDbCachingRemoteDirectoryInstanceLoader extends DbCachingRemoteDirectoryInstanceLoader {

    public ConfigurableDbCachingRemoteDirectoryInstanceLoader(DirectoryInstanceLoader remoteDirectoryInstanceLoader,
                                                              InternalDirectoryInstanceLoader internalDirectoryInstanceLoader,
                                                              DirectoryCacheFactory directoryCacheFactory,
                                                              CacheRefresherFactory cacheRefresherFactory,
                                                              AuditService auditService,
                                                              AuditLogUserMapper auditLogUserMapper,
                                                              AuditLogGroupMapper auditLogGroupMapper,
                                                              EventPublisher eventPublisher,
                                                              DirectoryDao directoryDao) {
        super(remoteDirectoryInstanceLoader, internalDirectoryInstanceLoader,
                directoryCacheFactory, cacheRefresherFactory, auditService, auditLogUserMapper, auditLogGroupMapper,
                eventPublisher, directoryDao);
    }

    @Override
    public RemoteDirectory getDirectory(Directory directory) throws DirectoryInstantiationException {
        if (Boolean.parseBoolean(directory.getAttributes().get(DirectoryProperties.CACHE_ENABLED))) {
            return super.getDirectory(directory);
        } else {
            final RemoteDirectory uncachedDirectory = getRawDirectory(
                    directory.getId(),
                    directory.getImplementationClass(),
                    directory.getAttributes()
            );
            if (auditService.isEnabled()) {
                return new AuditingDirectoryDecorator(
                        uncachedDirectory,
                        auditService,
                        auditLogUserMapper,
                        auditLogGroupMapper,
                        directory.getName());
            } else {
                return uncachedDirectory;
            }
        }
    }
}
