package com.atlassian.crowd.crypto;

import com.atlassian.crowd.embedded.api.Encryptor;

import java.util.UUID;

/**
 * Encryptor that delegates another Encryptor and adds to them the salting feature.
 * <p>
 * The implementation works correctly with decrypting both salted and non-salted passwords,
 * so after enabling no migration of the existing passwords will be needed. Nevertheless,
 * new encrypted passwords will be salted.
 */
public class SaltingEncryptor implements Encryptor {
    static final String SALT_SEPARATOR = "|SALT-";

    private static final int UUID_LENGTH = 36;
    private static final int SALT_LENGTH = UUID_LENGTH + SALT_SEPARATOR.length();

    private final Encryptor delegate;

    public SaltingEncryptor(Encryptor delegate) {
        this.delegate = delegate;
    }

    @Override
    public String encrypt(String password) {
        return delegate.encrypt(addSalt(password));
    }

    @Override
    public String decrypt(String encryptedPassword) {
        return removeSaltIfNeeded(delegate.decrypt(encryptedPassword));
    }

    private String addSalt(String password) {
        if (password == null) {
            return null;
        }
        return password + SALT_SEPARATOR + UUID.randomUUID();
    }

    private String removeSaltIfNeeded(String text) {
        if (text == null) {
            return null;
        }
        final int expectedSaltStart = text.length() - SALT_LENGTH;
        if (expectedSaltStart >= 0 && text.startsWith(SALT_SEPARATOR, expectedSaltStart)) {
            return text.substring(0, expectedSaltStart);
        }
        return text;
    }

    @Override
    public boolean changeEncryptionKey() {
        return delegate.changeEncryptionKey();
    }
}
