package com.atlassian.crowd.crypto;

import com.atlassian.crowd.embedded.api.Encryptor;
import com.atlassian.crowd.embedded.api.SwitchableEncryptor;
import com.atlassian.crowd.manager.property.EncryptionSettings;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;

public class PrefixBasedSwitchableEncryptor implements SwitchableEncryptor {
    private final Logger log = LoggerFactory.getLogger(PrefixBasedSwitchableEncryptor.class);

    private final EncryptionSettings encryptionSettings;
    private final Map<String, Encryptor> encryptors;

    public PrefixBasedSwitchableEncryptor(EncryptionSettings encryptionSettings, Map<String, Encryptor> encryptors) {
        this.encryptionSettings = encryptionSettings;
        this.encryptors = encryptors;
    }

    static String addPrefix(String password, String prefix) {
        return "{" + prefix + "}" + password;
    }

    @Override
    public String encrypt(String password) {
        Optional<String> defaultEncryptor = encryptionSettings.getDefaultEncryptor();
        if (!defaultEncryptor.isPresent()) {
            return password;
        }

        Encryptor encryptor = getEncryptor(defaultEncryptor.get());

        if (encryptor == null) {
            log.warn("Default encryptor not found for key {}, encryption not performed. " +
                    "Please select one of existing keys {}", defaultEncryptor, encryptors.keySet());
            return password;
        }

        log.debug("Encrypting password with encryptor {}", defaultEncryptor.get());
        return addPrefix(encryptor.encrypt(password), defaultEncryptor.get());
    }

    protected Encryptor getEncryptor(String encryptorKey) {
        return encryptors.get(encryptorKey);
    }

    @Override
    public String decrypt(String encryptedPassword) {
        int closeIndex = encryptedPassword.indexOf('}');
        if (closeIndex > 0 && encryptedPassword.charAt(0) == '{') {
            String encryptorKey = encryptedPassword.substring(1, closeIndex);
            Encryptor encryptor = getEncryptor(encryptorKey);
            if (encryptor != null) {
                try {
                    return encryptor.decrypt(encryptedPassword.substring(closeIndex + 1));
                } catch (RuntimeException e) {
                    log.error("Error during decryption", e);
                    return encryptedPassword;
                }
            } else {
                log.error("Password encrypted with unknown encryptor {}. Cannot decrypt password. ", encryptorKey);
            }
        } else {
            log.debug("Cannot detected encryptor. Decryption not performed. Returned encrypted password");
        }
        return encryptedPassword;
    }

    @Override
    public void switchEncryptor(String encryptorKey) {
        if (encryptorKey == null) {
            encryptionSettings.setDefaultEncryptor(null);
            log.info("Switched to non-encrypting/decrypting encryptor. Encryption is now disabled");
            return;
        }

        Preconditions.checkArgument(encryptors.containsKey(encryptorKey), "Unknown encryptor %s", encryptorKey);

        Encryptor encryptor = getEncryptor(encryptorKey);
        encryptor.changeEncryptionKey();
        encryptionSettings.setDefaultEncryptor(encryptorKey);
        log.info("Switched to encryptor with key: {}", encryptorKey);
    }

    @Override
    public boolean changeEncryptionKey() {
        Optional<String> defaultEncryptor = encryptionSettings.getDefaultEncryptor();
        Preconditions.checkArgument(defaultEncryptor.isPresent(), "Cannot rotate encryption key when encryption is disabled");
        Encryptor encryptor = getEncryptor(defaultEncryptor.get());
        Preconditions.checkArgument(encryptor != null, "Unsupported encryptor selected. Key rotation is possible only for supported encryptors");
        boolean keyChangeSupported = encryptor.changeEncryptionKey();
        Preconditions.checkArgument(keyChangeSupported, "Encryptor %s does not support key rotation", defaultEncryptor.get());
        return true;
    }

    @Override
    public Collection<String> getAvailableEncryptorKeys() {
        return encryptors.keySet();
    }

    @Override
    public Optional<String> getCurrentEncryptorKey() {
        return encryptionSettings.getDefaultEncryptor();
    }
}
