package com.atlassian.crowd.crypto;

import com.atlassian.crowd.common.properties.EncryptionProperties;
import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Set;
import java.util.stream.Collectors;

import static java.nio.file.attribute.PosixFilePermission.OWNER_EXECUTE;
import static java.nio.file.attribute.PosixFilePermission.OWNER_READ;
import static java.nio.file.attribute.PosixFilePermission.OWNER_WRITE;

public class EncryptionKeyFilePermissionChanger {
    private static final ImmutableSet<PosixFilePermission> POSIX_OWNER_PERMISSIONS = ImmutableSet.of(OWNER_READ, OWNER_WRITE, OWNER_EXECUTE);
    private static final Logger log = LoggerFactory.getLogger(EncryptionKeyFilePermissionChanger.class);

    public void makeFileReadableOnlyByOwner(String keyPath) {
        if (!EncryptionProperties.SET_ENCRYPTION_KEYS_OWNERSHIP_ATTRIBUTES.getValue()) {
            return;
        }

        try {
            Path path = Paths.get(keyPath);
            PosixFileAttributeView posixFileAttributeView = Files.getFileAttributeView(path, PosixFileAttributeView.class);
            if (posixFileAttributeView != null) {
                removeNonOwnerPosixAttributes(posixFileAttributeView, path);
                return;
            }

            log.warn("Cannot set access attributes of keyFile {}", keyPath);
        } catch (IOException e) {
            log.error("Cannot set access attributes of keyFile {}", keyPath, e);
        }
    }

    private void removeNonOwnerPosixAttributes(PosixFileAttributeView posixFileAttributeView, Path path) throws IOException {
        Set<PosixFilePermission> permissions = posixFileAttributeView.readAttributes().permissions();
        if (!permissions.contains(OWNER_READ)) {
            log.warn("Will skip permissions update for {}, because file would be not readable by owner", path);
            return;
        }

        Set<PosixFilePermission> updatedPermissions = permissions.stream()
                .filter(POSIX_OWNER_PERMISSIONS::contains)
                .collect(Collectors.toSet());

        posixFileAttributeView.setPermissions(updatedPermissions);
    }
}
