package com.atlassian.crowd.crypto;

import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.embedded.api.Encryptor;
import com.atlassian.crowd.embedded.api.SwitchableEncryptor;
import com.atlassian.crowd.lock.ClusterLockWrapper;
import com.atlassian.crowd.manager.crypto.DefaultEncryptionManager;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Optional;

/**
 * {@link Encryptor} that synchronizes on given cluster lock when doing encryption.
 */
public class ClusterLockingEncryptor implements SwitchableEncryptor {
    private final SwitchableEncryptor delegate;
    private final ClusterLockWrapper clusterLockWrapper;

    public ClusterLockingEncryptor(SwitchableEncryptor delegate, ClusterLockService clusterLockService) {
        this.delegate = delegate;
        this.clusterLockWrapper = new ClusterLockWrapper(() -> clusterLockService.getLockForName(DefaultEncryptionManager.LOCK_NAME));
    }

    @Override
    public String encrypt(String password) {
        return clusterLockWrapper.run(() -> delegate.encrypt(password));
    }

    @Override
    public String decrypt(String encryptedPassword) {
        return delegate.decrypt(encryptedPassword);
    }

    @Override
    public boolean changeEncryptionKey() {
        return clusterLockWrapper.run(delegate::changeEncryptionKey);
    }

    @Override
    public void switchEncryptor(@Nullable String encryptorKey) {
        clusterLockWrapper.run(() -> delegate.switchEncryptor(encryptorKey));
    }

    @Override
    public Collection<String> getAvailableEncryptorKeys() {
        return delegate.getAvailableEncryptorKeys();
    }

    @Override
    public Optional<String> getCurrentEncryptorKey() {
        return delegate.getCurrentEncryptorKey();
    }
}
