package com.atlassian.crowd.core.event;

import com.atlassian.event.internal.AsynchronousAbleEventDispatcher;
import com.atlassian.event.spi.EventExecutorFactory;
import com.atlassian.event.spi.ListenerInvoker;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

/**
 * Wraps dispatching each atlassian-events event in a separate hibernate transaction.
 *
 * These are marked as PROPAGATION_REQUIRES_NEW, to get a new synchronisation context, and to work correclty
 * with being invoked when invoked in an afterCommit() call by {@link TransactionAwareEventPublisher}.
 * The physical transaction for the calling thread should actually never be active when
 * {{@link #dispatch(ListenerInvoker, Object)}} is called.
 */
public class TransactionAwareEventDispatcher extends AsynchronousAbleEventDispatcher {
    /**
     * Listeners with this scope will not be automatically run in transaction and will be responsible of transaction
     * management on it's own.
     */
    public static final String DISABLE_TRANSACTION = "DISABLE_TRANSACTION";
    private final TransactionTemplate transactionTemplate;

    public TransactionAwareEventDispatcher(EventExecutorFactory executorFactory, PlatformTransactionManager transactionManager) {
        super(executorFactory);
        this.transactionTemplate = new TransactionTemplate(transactionManager,
                new DefaultTransactionDefinition(DefaultTransactionDefinition.PROPAGATION_REQUIRES_NEW));
    }

    @Override
    public void dispatch(ListenerInvoker invoker, Object event) {
        if (invoker.getScope().map(DISABLE_TRANSACTION::equals).orElse(false)) {
            super.dispatch(invoker, event);
            return;
        }
        transactionTemplate.execute(status -> {
            super.dispatch(invoker, event);
            return null;
        });
    }
}
