package com.atlassian.crowd.cache;

import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;

import com.atlassian.cache.Cache;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.model.user.User;

import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.ObjectWriter;

/**
 * Default implementation of UserAuthorisationCache in Crowd.
 *
 * @since v2.2
 */
public class UserAuthorisationCacheImpl implements UserAuthorisationCache {
    private static final ObjectWriter jsonWriter = new ObjectMapper().writer();
    private final Cache<String, Boolean> cache;

    public UserAuthorisationCacheImpl(Cache<String, Boolean> cache) {
        this.cache = cache;
    }

    public void setPermitted(final User user, final String applicationName, final boolean permitted) {
        cache.put(getCacheKey(user, applicationName), permitted);
    }

    public Boolean isPermitted(final User user, final String applicationName) {
        return cache.get(getCacheKey(user, applicationName));
    }

    public void clear() {
        cache.removeAll();
    }

    /**
     * Returns the cache key based on the username, directory id, external id and application name.
     * The key MUST be unique enough to produce distinct entries for user renames,
     * users being shadowed by removals and creations, or directory order changes.
     * Use a <code>String</code> to allow use with any cache implementation, rather than an in-memory Java one.
     *
     * @param user            the user to build the key for
     * @param applicationName name of the application
     * @return cache key
     */
    static String getCacheKey(final User user, final String applicationName) {
        try {
            final List<? extends Serializable> cacheKeyComponents = Arrays.asList(
                    IdentifierUtils.toLowerCase(user.getName()),
                    user.getDirectoryId(),
                    user.getExternalId(),
                    IdentifierUtils.toLowerCase(applicationName));
            return jsonWriter.writeValueAsString(cacheKeyComponents);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
