/**
 * Stickytable headers for confluence
 *
 * Adds stalking headers for viewpage and editor dependeds on a modified version of stalky headers
 * that allow extra options to be passed
 */
define("sticky-table-headers", [
    "ajs",
    "sticky-table-headers/viewpage",
    "sticky-table-headers/editor",
    "skate"
], function(
    AJS,
    StickyViewpage,
    StickyEditor,
    skate
) {
    "use strict";

    var retryScheduled = false;
    var initialization = AJS.debounce(StickyViewpage.initialize, 0);

    /**
     * Initialize table until we seen a thead
     * All the logic in initialize is depending on thead tag
     * However it is converted from th by sortable table asynchronously
     * There is a chance that we will have race condition making the header not proper initialized.
     *
     * @param el Table element
     * @param retry Maximum times to retry
     */
    function initializeUntilTheadAppears(el, retry) {
        if (retry < 0) {
            return;
        }

        if (el.getElementsByTagName('thead').length === 0) {
            if (!retryScheduled) {
              setTimeout(initializeUntilTheadAppears.bind(null, el, --retry), 50);
              retryScheduled = true;
            }
        } else {
            initialization();
            retryScheduled = false;
        }
    }

    return {
        initialize: function() {
            if (!AJS.DarkFeatures.isEnabled("confluence-table-enhancements.sticky-headers.disabled")) {
                // SPA compatible
                skate('confluenceTable', {
                    type: skate.types.CLASS,
                    attached: function (el) {
                        // Initialize until the table has a thead
                        initializeUntilTheadAppears(el, 5);
                    }
                });
            }

            if (AJS.DarkFeatures.isEnabled("confluence-table-enhancements.sticky-headers-editor")) {
                // auto initialize for the editor
                AJS.bind("rte-ready", StickyEditor.initialize);

                // this is to make stickyheaders works with quick-edit
                AJS.bind("rte-quick-edit-push-state", StickyEditor.initialize);
                AJS.bind("rte-quick-edit-push-hash", StickyEditor.initialize);
            }
        }
    };


});

// auto start
require("sticky-table-headers").initialize();
