var RYQ = RYQ || {};

RYQ.QuickNavEntry = Backbone.Model.extend({
    classNameByType: {
        "blogpost": "content-type-blogpost",
        "page": "content-type-page"
    },

    parse: function (page) {
        return {
            className: this.classNameByType[page["type"]],
            name: page["title"],
            href: AJS.contextPath() + page["url"],
            id: page["id"],
            spaceName: page["space"],
            lastSeen: page["lastSeen"]
        }
    }
}, {

    /**
     * Escapes the name and spaceName attribute of an array of QuickNavEntry json objects.
     *
     * @param objects
     * @returns {*}
     */
    escape: function(objects) {
        var clones = _.map(objects, _.clone);

        _.each(clones, function (obj) {
            obj.name = _.escape(obj.name);
            obj.spaceName = _.escape(obj.spaceName);
        });

        return clones;
    }
});

RYQ.QuickNavEntryCollection = Backbone.Collection.extend({
    model: RYQ.QuickNavEntry,
    url: AJS.contextPath() + "/rest/recentlyviewed/1.0/recent?limit=8",

    /**
     * Searches all pages for a title that matches query. Triggers a 'filter' event with an array of pages that match
     * the query.
     *
     * @param query
     */
    search: function(query) {
        var result;
        if (AJS.$.trim(query).length === 0) {
            // empty search shows everything
            result = this.models;
        } else {
            /**
             * search algorithm:
             * - split search string into separate words
             * - try to find each word in the combination of page+space
             * - return a result if each word was found
             */
            var words = query.match(/[^\s-]+/g);
            result = this.filter(function (page) {
                var title = page.get('name');
                var haystack = title.toLowerCase();

                return _.all(words, function(word) {
                    return haystack.indexOf(word.toLowerCase()) !== -1;
                });

            });
        }

        this.trigger("filter", result);
        return result;
    },

    comparator: function(page) {
        return -(page.get('lastSeen'));
    }
});

(function ($) {
    RYQ.QuickNav = Backbone.View.extend({

        initialize: function() {
            this.moreLink = {
                className: "recently-viewed",
                href: "#",
                name: AJS.I18n.getText("com.atlassian.confluence.plugins.recently.viewed.quicknav.more")
            };
            this.$input = $("#quick-search-query");

            this.makeDropdown();
            this.addShowHideHandlers();
            this.getHistory = _.once(this._getHistory);

            _.bindAll(this, 'makeDropdown', 'addSearchResultBoostingHandler', '_getHistory', 'render',
                'addShowHideHandlers', '_getItemsToShow', 'showQuickResults', 'onQuickSearch');
        },

        makeDropdown: function() {
            var addSpaceName = function(dd) {
                $("a", dd).each(function () {
                    var $a = $(this);
                    var $span = $a.find("span");
                    // get the hidden space name property from the span
                    var properties = $span.data("properties");
                    var spaceName = properties ? properties.spaceName : null;
                    if (spaceName && !$a.is(".content-type-spacedesc")) {
                        // clone the original link for now. This could potentially link to the space?
                        $a.after($a.clone().attr("class", "space-name").html(spaceName));
                        // add another class so we can restyle to make room for the space name
                        $a.parent().addClass("with-space-name");
                    }
                });
            };

            var self = this;
            this.$dropdown = AJS.inputDrivenDropdown({
                dropdownPlacement: function (dropDown) {
                    self.$input.closest("form").find(".quick-nav-drop-down").append(dropDown);
                },
                dropdownPostprocess: function (dd) {
                    addSpaceName(dd);
                },
                ajsDropDownOptions: {
                    className: "recently-viewed-dropdown"
                }
            });
        },

        // merge search results with recently viewed
        addSearchResultBoostingHandler: function() {
            var self = this;
            $(window).on("quicksearch.ajax-success", function(e, data) {
                // depending on the confluence version quicksearch hits different URLs
                var isOldQuickSearch = data.url.match("/json/contentnamesearch.action"); // pre 5.2
                var isNewQuickSearch = data.url.match(/rest\/quicknav\/\d\/search/); // post 5.2
                if (isOldQuickSearch || isNewQuickSearch) {
                    self.onQuickSearch(e, data);
                }
            });
        },

        _getHistory: function() {
            return this.collection.fetch().done(this.addSearchResultBoostingHandler);
        },

        render: function() {
            // Do not render recent panel if in anonymous mode
            if (!AJS.Meta.get('remote-user')) {
                return;
            }

            var dfd = this.getHistory();
            dfd.done(_.bind(function () {
                if (AJS.dropDown.current == null && this.collection.length !== 0 && this.$input.val().length === 0) {
                    this.showQuickResults();
                }
            }, this));

            // toggle loading spinner
            var $input = this.$input;
            $input.trigger("quick-search-loading-start");
            dfd.always(function () {
                $input.trigger("quick-search-loading-stop");
            });
        },

        /**
         * Shows the dropdown when the input is empty and hides it if its not. Shows the dropdown on focus and click.
         */
        addShowHideHandlers: function() {
            var self = this;

            this.$input
                .on("focus",function () {
                    self.render();
                })
                .on("click", function (e) {
                    // this is a workaround to avoid the dropdown being hidden when you click into the input field after
                    // it has acquired focus via click.
                    e.stopPropagation();
                    self.render();
                })
                .on("keyup", function (e) {
                    var keyEscape = e.which === 27;
                    var keyEnter = e.which === 13;
                    var inputNotEmpty = $(this).val().length !== 0;

                    if (inputNotEmpty || keyEscape) {
                        if (self.$dropdown.dd && self.$dropdown.dd.$.is(":visible")) {
                            self.$dropdown.hide();
                        }
                    } else if (!keyEnter) {
                        self.render();
                    }
                });
        },

        _getItemsToShow: function () {
            var items = this.collection.toJSON();

            // remove most recent page from list if it's the same as current page
            var mostRecentPageIsCurrentPage = items.length > 0 && items[0].id == AJS.Meta.get("page-id");
            if (mostRecentPageIsCurrentPage) {
                items.shift();
            }

            return items;
        },

        showQuickResults: function () {
            var items = RYQ.QuickNavEntry.escape(this._getItemsToShow());
            this.$dropdown.show([ items, [this.moreLink]], "", "");

            $(".recently-viewed-dropdown").on("click", ".recently-viewed", function (e) {
                e.preventDefault();
                $("#view-user-history-link").click();
                RYQ.util.analytics.trackQuickNavRecentlyDialogOpen();
            });

            $(".recently-viewed-dropdown").on("click", ".with-space-name", function (e) {
                RYQ.util.analytics.trackQuickNavPageOpen();
            });

            RYQ.util.analytics.trackQuickNavOpen();
        },

        onQuickSearch: function(e, data) {
            var query = data.json.query;
            var filteredPages = _.map(this.collection.search(query), function(obj) { return obj.attributes; });
            filteredPages = RYQ.QuickNavEntry.escape(filteredPages);

            if (filteredPages.length == 0) {
                return;
            }

            var content = data.json.contentNameMatches;
            var categoryIndex = -1;
            for (var i=0; i < content.length; i++) {
                var firstElementOfTypeClassName = content[i][0].className;
                if (firstElementOfTypeClassName == "content-type-blogpost" || firstElementOfTypeClassName == "content-type-page") {
                    categoryIndex = i;
                    break;
                }
            }

            if (categoryIndex != -1) {
                var resultPages = content[categoryIndex];
                var recentlyNumber = Math.min(resultPages.length > 4 ? 2 : 6 - resultPages.length, filteredPages.length);
                resultPages.unshift.apply(resultPages, _(filteredPages).first(recentlyNumber));
                content[categoryIndex] = _.uniq(resultPages, function(page) { return +page.id; });

                if (resultPages.length > 6) {
                    var pageToRemove = 6 - recentlyNumber;
                    for (var j=pageToRemove; j > 0; j--) {
                        resultPages.pop();
                    }
                }
            } else {
                content.unshift(_(filteredPages).first(6));
            }
        }
    });
}(AJS.$));
